-- =============================================
-- Multi-Tenant WhatsApp Survey Bot Database
-- MariaDB/MySQL Compatible Schema
-- Full Feature Set with AI Integration
-- =============================================

-- Set proper UTF-8 encoding for emoji support
SET NAMES utf8mb4 COLLATE utf8mb4_unicode_ci;
SET CHARACTER SET utf8mb4;

-- Sessions table (required for auth)
CREATE TABLE IF NOT EXISTS sessions (
  session_id VARCHAR(128) PRIMARY KEY,
  expires INT UNSIGNED NOT NULL,
  data MEDIUMTEXT,
  INDEX idx_expires (expires)
);

-- Admin users table
CREATE TABLE IF NOT EXISTS admin_users (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  username VARCHAR(50) UNIQUE NOT NULL,
  password VARCHAR(255) NOT NULL,
  email VARCHAR(100),
  is_super_admin BOOLEAN DEFAULT FALSE,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Tenants table (multi-tenant support)
CREATE TABLE IF NOT EXISTS tenants (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  name VARCHAR(255) NOT NULL,
  slug VARCHAR(100) UNIQUE NOT NULL,
  description TEXT,
  contact_email VARCHAR(100),
  contact_phone VARCHAR(20),
  website VARCHAR(255),
  logo_url VARCHAR(255),
  is_active BOOLEAN DEFAULT TRUE,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Tenant users table
CREATE TABLE IF NOT EXISTS tenant_users (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  tenant_id VARCHAR(36) NOT NULL,
  username VARCHAR(50) NOT NULL,
  password VARCHAR(255) NOT NULL,
  email VARCHAR(100),
  role ENUM('admin', 'manager', 'viewer') DEFAULT 'viewer',
  is_active BOOLEAN DEFAULT TRUE,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  FOREIGN KEY (tenant_id) REFERENCES tenants(id) ON DELETE CASCADE,
  UNIQUE KEY unique_tenant_username (tenant_id, username)
);

-- WhatsApp users table
CREATE TABLE IF NOT EXISTS users (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  tenant_id VARCHAR(36) NOT NULL,
  phone_number VARCHAR(20) NOT NULL,
  chat_id VARCHAR(50) NOT NULL,
  name VARCHAR(255),
  profile_name VARCHAR(255),
  language ENUM('ms', 'en', 'zh', 'ta') DEFAULT 'ms',
  is_active BOOLEAN DEFAULT TRUE,
  last_activity TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  FOREIGN KEY (tenant_id) REFERENCES tenants(id) ON DELETE CASCADE,
  UNIQUE KEY unique_tenant_chat (tenant_id, chat_id),
  INDEX idx_phone_number (phone_number),
  INDEX idx_chat_id (chat_id),
  INDEX idx_last_activity (last_activity)
);

-- Surveys table
CREATE TABLE IF NOT EXISTS surveys (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  tenant_id VARCHAR(36) NOT NULL,
  title VARCHAR(255) NOT NULL,
  description TEXT,
  is_active BOOLEAN DEFAULT TRUE,
  created_by VARCHAR(36),
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  FOREIGN KEY (tenant_id) REFERENCES tenants(id) ON DELETE CASCADE,
  INDEX idx_tenant_active (tenant_id, is_active)
);

-- Conversations table
CREATE TABLE IF NOT EXISTS conversations (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  user_id VARCHAR(36) NOT NULL,
  tenant_id VARCHAR(36) NOT NULL,
  survey_id VARCHAR(36),
  title VARCHAR(255),
  status ENUM('active', 'inactive', 'archived') DEFAULT 'active',
  language ENUM('ms', 'en', 'zh', 'ta') DEFAULT 'ms',
  last_message_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
  FOREIGN KEY (tenant_id) REFERENCES tenants(id) ON DELETE CASCADE,
  FOREIGN KEY (survey_id) REFERENCES surveys(id) ON DELETE SET NULL,
  INDEX idx_user_status (user_id, status),
  INDEX idx_tenant_status (tenant_id, status),
  INDEX idx_last_message (last_message_at)
);

-- Messages table
CREATE TABLE IF NOT EXISTS messages (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  conversation_id VARCHAR(36) NOT NULL,
  content TEXT NOT NULL,
  sender_type ENUM('user', 'bot') NOT NULL,
  message_type ENUM('text', 'image', 'document', 'interactive') DEFAULT 'text',
  metadata JSON,
  processed_at TIMESTAMP NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (conversation_id) REFERENCES conversations(id) ON DELETE CASCADE,
  INDEX idx_conversation_type (conversation_id, sender_type),
  INDEX idx_created_at (created_at)
);

-- Survey responses table (AI categorization)
CREATE TABLE IF NOT EXISTS survey_responses (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  conversation_id VARCHAR(36) NOT NULL,
  survey_id VARCHAR(36) NOT NULL,
  user_id VARCHAR(36) NOT NULL,
  tenant_id VARCHAR(36) NOT NULL,
  intent_category ENUM('high_potential', 'medium_potential', 'low_potential', 'not_eligible', 'unclear') DEFAULT 'unclear',
  business_interest_score INT DEFAULT 0,
  urgency_level ENUM('immediate', 'soon', 'future', 'none') DEFAULT 'none',
  eligibility_status ENUM('eligible', 'maybe_eligible', 'not_eligible', 'unknown') DEFAULT 'unknown',
  response_data JSON,
  analyzed_at TIMESTAMP NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  FOREIGN KEY (conversation_id) REFERENCES conversations(id) ON DELETE CASCADE,
  FOREIGN KEY (survey_id) REFERENCES surveys(id) ON DELETE CASCADE,
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
  FOREIGN KEY (tenant_id) REFERENCES tenants(id) ON DELETE CASCADE,
  INDEX idx_tenant_category (tenant_id, intent_category),
  INDEX idx_survey_status (survey_id, eligibility_status)
);

-- Demographics table (AI extraction)
CREATE TABLE IF NOT EXISTS demographic_data (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  user_id VARCHAR(36) NOT NULL,
  tenant_id VARCHAR(36) NOT NULL,
  age_range VARCHAR(20),
  gender ENUM('male', 'female', 'other', 'unknown') DEFAULT 'unknown',
  race VARCHAR(50),
  occupation VARCHAR(100),
  location VARCHAR(100),
  income_range VARCHAR(50),
  education_level VARCHAR(50),
  family_size INT,
  extracted_from_messages BOOLEAN DEFAULT TRUE,
  confidence_score DECIMAL(3,2) DEFAULT 0.00,
  last_updated TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
  FOREIGN KEY (tenant_id) REFERENCES tenants(id) ON DELETE CASCADE,
  UNIQUE KEY unique_user_demographics (user_id),
  INDEX idx_tenant_demographics (tenant_id),
  INDEX idx_age_gender (age_range, gender)
);

-- Keywords table
CREATE TABLE IF NOT EXISTS keywords (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  tenant_id VARCHAR(36) NOT NULL,
  survey_id VARCHAR(36),
  keyword VARCHAR(255) NOT NULL,
  category ENUM('general', 'product', 'service', 'support', 'pricing', 'complaint', 'compliment') DEFAULT 'general',
  response_text TEXT,
  language ENUM('ms', 'en', 'zh', 'ta') DEFAULT 'ms',
  is_active BOOLEAN DEFAULT TRUE,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  FOREIGN KEY (tenant_id) REFERENCES tenants(id) ON DELETE CASCADE,
  FOREIGN KEY (survey_id) REFERENCES surveys(id) ON DELETE SET NULL,
  UNIQUE KEY unique_tenant_keyword (tenant_id, keyword, language),
  INDEX idx_tenant_category (tenant_id, category),
  INDEX idx_keyword_search (keyword)
);

-- Keyword mentions table
CREATE TABLE IF NOT EXISTS keyword_mentions (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  tenant_id VARCHAR(36) NOT NULL,
  survey_id VARCHAR(36),
  keyword VARCHAR(255) NOT NULL,
  count INT DEFAULT 1,
  channel ENUM('whatsapp', 'telegram', 'web') DEFAULT 'whatsapp',
  last_mentioned TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (tenant_id) REFERENCES tenants(id) ON DELETE CASCADE,
  FOREIGN KEY (survey_id) REFERENCES surveys(id) ON DELETE SET NULL,
  INDEX idx_tenant_keyword (tenant_id, keyword),
  INDEX idx_count_desc (count DESC),
  INDEX idx_last_mentioned (last_mentioned)
);

-- Customer leads table
CREATE TABLE IF NOT EXISTS customer_leads (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  tenant_id VARCHAR(36) NOT NULL,
  user_id VARCHAR(36) NOT NULL,
  survey_id VARCHAR(36),
  lead_score INT DEFAULT 0,
  status ENUM('new', 'contacted', 'qualified', 'converted', 'lost') DEFAULT 'new',
  source VARCHAR(50) DEFAULT 'whatsapp',
  contact_preference ENUM('whatsapp', 'call', 'email') DEFAULT 'whatsapp',
  notes TEXT,
  follow_up_date DATE,
  assigned_to VARCHAR(36),
  converted_at TIMESTAMP NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  FOREIGN KEY (tenant_id) REFERENCES tenants(id) ON DELETE CASCADE,
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
  FOREIGN KEY (survey_id) REFERENCES surveys(id) ON DELETE SET NULL,
  INDEX idx_tenant_status (tenant_id, status),
  INDEX idx_lead_score (lead_score DESC),
  INDEX idx_follow_up (follow_up_date)
);

-- Business templates table
CREATE TABLE IF NOT EXISTS business_templates (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  name VARCHAR(255) NOT NULL,
  category VARCHAR(100) NOT NULL,
  description TEXT,
  icon VARCHAR(10) CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci,
  prompts JSON,
  keywords JSON,
  is_active BOOLEAN DEFAULT TRUE,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  UNIQUE KEY unique_template_name (name),
  INDEX idx_category (category)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Global commands table (anti-hallucination commands)
CREATE TABLE IF NOT EXISTS global_commands (
  id VARCHAR(50) PRIMARY KEY,
  command VARCHAR(255) NOT NULL,
  response TEXT NOT NULL,
  category ENUM('system', 'identity', 'business', 'restriction', 'encouragement', 'response', 'emergency', 'help', 'explanation') DEFAULT 'system',
  priority INT DEFAULT 1,
  is_active BOOLEAN DEFAULT TRUE,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  UNIQUE KEY unique_command (command),
  INDEX idx_category_priority (category, priority),
  INDEX idx_active (is_active)
);

-- Keyword responses table
CREATE TABLE IF NOT EXISTS keyword_responses (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  template_id VARCHAR(36),
  keyword VARCHAR(255) NOT NULL,
  response TEXT NOT NULL,
  category ENUM('general', 'product', 'pricing', 'delivery', 'support') DEFAULT 'general',
  language ENUM('ms', 'en', 'zh', 'ta') DEFAULT 'ms',
  is_active BOOLEAN DEFAULT TRUE,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  FOREIGN KEY (template_id) REFERENCES business_templates(id) ON DELETE CASCADE,
  INDEX idx_template_keyword (template_id, keyword),
  INDEX idx_keyword_language (keyword, language)
);

-- WhatsApp providers table
CREATE TABLE IF NOT EXISTS whatsapp_providers (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  tenant_id VARCHAR(36) NOT NULL,
  provider_name ENUM('green_api', 'getitsms', 'meta_api') NOT NULL,
  instance_id VARCHAR(100),
  access_token VARCHAR(255),
  webhook_url VARCHAR(255),
  is_active BOOLEAN DEFAULT FALSE,
  is_primary BOOLEAN DEFAULT FALSE,
  settings JSON,
  last_health_check TIMESTAMP NULL,
  health_status ENUM('healthy', 'warning', 'error') DEFAULT 'healthy',
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  FOREIGN KEY (tenant_id) REFERENCES tenants(id) ON DELETE CASCADE,
  UNIQUE KEY unique_tenant_provider (tenant_id, provider_name),
  INDEX idx_tenant_active (tenant_id, is_active)
);

-- Pricing plans table
CREATE TABLE IF NOT EXISTS pricing_plans (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  name VARCHAR(100) NOT NULL,
  description TEXT,
  price_monthly DECIMAL(10,2) DEFAULT 0.00,
  price_yearly DECIMAL(10,2) DEFAULT 0.00,
  max_messages INT DEFAULT 1000,
  max_users INT DEFAULT 100,
  max_surveys INT DEFAULT 1,
  features JSON,
  is_active BOOLEAN DEFAULT TRUE,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  UNIQUE KEY unique_plan_name (name)
);

-- Payment gateways table
CREATE TABLE IF NOT EXISTS payment_gateways (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  name VARCHAR(100) NOT NULL,
  provider VARCHAR(50) NOT NULL,
  api_key VARCHAR(255),
  secret_key VARCHAR(255),
  webhook_url VARCHAR(255),
  is_active BOOLEAN DEFAULT FALSE,
  is_sandbox BOOLEAN DEFAULT TRUE,
  supported_methods JSON,
  settings JSON,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  UNIQUE KEY unique_gateway_name (name)
);

-- Tenant subscriptions table
CREATE TABLE IF NOT EXISTS tenant_subscriptions (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  tenant_id VARCHAR(36) NOT NULL,
  pricing_plan_id VARCHAR(36) NOT NULL,
  status ENUM('active', 'cancelled', 'expired', 'pending') DEFAULT 'pending',
  starts_at TIMESTAMP NOT NULL,
  expires_at TIMESTAMP NOT NULL,
  payment_gateway_id VARCHAR(36),
  subscription_data JSON,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  FOREIGN KEY (tenant_id) REFERENCES tenants(id) ON DELETE CASCADE,
  FOREIGN KEY (pricing_plan_id) REFERENCES pricing_plans(id) ON DELETE RESTRICT,
  FOREIGN KEY (payment_gateway_id) REFERENCES payment_gateways(id) ON DELETE SET NULL,
  INDEX idx_tenant_status (tenant_id, status),
  INDEX idx_expiry (expires_at)
);

-- Files table (for document management)
CREATE TABLE IF NOT EXISTS files (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  tenant_id VARCHAR(36) NOT NULL,
  filename VARCHAR(255) NOT NULL,
  original_name VARCHAR(255) NOT NULL,
  file_path VARCHAR(500) NOT NULL,
  file_size BIGINT NOT NULL,
  mime_type VARCHAR(100),
  file_type ENUM('image', 'document', 'audio', 'video', 'other') DEFAULT 'other',
  uploaded_by VARCHAR(36),
  is_processed BOOLEAN DEFAULT FALSE,
  processed_at TIMESTAMP NULL,
  metadata JSON,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (tenant_id) REFERENCES tenants(id) ON DELETE CASCADE,
  INDEX idx_tenant_type (tenant_id, file_type),
  INDEX idx_uploaded_by (uploaded_by)
);

-- System logs table
CREATE TABLE IF NOT EXISTS system_logs (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  tenant_id VARCHAR(36),
  user_id VARCHAR(36),
  level ENUM('info', 'warning', 'error', 'debug') DEFAULT 'info',
  category VARCHAR(50) NOT NULL,
  message TEXT NOT NULL,
  data JSON,
  ip_address VARCHAR(45),
  user_agent TEXT,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (tenant_id) REFERENCES tenants(id) ON DELETE SET NULL,
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE SET NULL,
  INDEX idx_tenant_level (tenant_id, level),
  INDEX idx_category_created (category, created_at),
  INDEX idx_created_at (created_at)
);

-- System settings table
CREATE TABLE IF NOT EXISTS system_settings (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  setting_key VARCHAR(100) UNIQUE NOT NULL,
  setting_value TEXT,
  description TEXT,
  is_public BOOLEAN DEFAULT FALSE,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  INDEX idx_key_public (setting_key, is_public)
);

-- =============================================
-- INSERT DEFAULT DATA
-- =============================================

-- Default tenant
INSERT IGNORE INTO tenants (id, name, slug, description, contact_email, is_active) VALUES
('a1ffbb8c-cb66-43de-8d04-175289124254', 'Default Tenant', 'default', 'Default system tenant', 'admin@system.com', TRUE);

-- Default admin user (password: ##Master8899)
-- Hash generated using bcrypt with salt rounds 10
INSERT IGNORE INTO admin_users (id, username, password, email, is_super_admin) VALUES
('admin-001', 'admin', '$2b$10$9DttV7tcfs1rxMSMOmoIsO8KDgiFvSd8AqkyxIKhLTAymx22tRmlu', 'admin@system.com', TRUE);

-- Default pricing plans
INSERT IGNORE INTO pricing_plans (name, description, price_monthly, max_messages, max_users, max_surveys) VALUES
('Freemium', 'Basic plan with limited features', 0.00, 1000, 100, 1),
('Starter', 'Perfect for small businesses', 29.00, 5000, 500, 3),
('Growth', 'Ideal for growing companies', 99.00, 20000, 2000, 10),
('Enterprise', 'Full-featured plan for large organizations', 299.00, 100000, 10000, 50);

-- Default system settings
INSERT IGNORE INTO system_settings (setting_key, setting_value, description, is_public) VALUES
('app_name', 'WhatsApp Survey Chatbot', 'Application name', TRUE),
('app_version', '1.0.0', 'Application version', TRUE),
('maintenance_mode', 'false', 'Maintenance mode status', FALSE),
('max_file_size', '10485760', 'Maximum file upload size in bytes', FALSE),
('default_tenant_id', 'a1ffbb8c-cb66-43de-8d04-175289124254', 'Default tenant ID', FALSE);

-- Business templates data (Complete 20+ templates with real prompts and keywords)
INSERT IGNORE INTO business_templates (id, name, category, description, icon, prompts, keywords, is_active) VALUES

-- Food & Beverage
('template-001', 'Restoran & Makanan', 'food', 'Template untuk restoran, kedai makan, dan food delivery', '🍽️',
 '{"system": "Anda adalah AI assistant untuk restoran. Anda membantu pelanggan dengan: Maklumat menu dan harga, Tempahan meja, Delivery dan takeaway, Waktu operasi, Promosi terkini. Selalu mesra, professional dan promosi makanan dengan cara yang menarik.", "company_info": "🍽️ **[NAMA RESTORAN]**\\n📍 Alamat: [ALAMAT LENGKAP]\\n📞 Telefon: [NOMBOR TELEFON]\\n⏰ Waktu: [WAKTU OPERASI]\\n🚗 Delivery: [YA/TIDAK]\\n💳 Pembayaran: [TUNAI/CARD/EWALLET]"}',
 '["menu", "harga", "delivery", "booking", "waktu", "makanan", "restoran", "order"]', TRUE),

-- Technology
('template-002', 'Alesa IT Services', 'technology', 'Template untuk Alesa IT Services - Peneraju AI Chatbot di Malaysia', '🤖',
 '{"system": "Anda adalah AI assistant untuk Alesa IT Services, peneraju teknologi AI Chatbot di Malaysia. Anda membantu pelanggan dengan: Pembangunan AI Chatbot custom, Integrasi WhatsApp Business API, Automasi perniagaan menggunakan AI, Konsultasi transformasi digital, Sokongan teknikal dan maintenance. Selalu professional, tunjukkan expertise AI/ML, dan fokus pada nilai ROI untuk klien.", "company_info": "🤖 **ALESA IT SERVICES**\\n🏆 Peneraju AI Chatbot Malaysia\\n📍 Kuala Lumpur, Malaysia\\n📞 Hotline: +6012-345-6789\\n💬 WhatsApp: +6012-345-6789\\n🌐 Web: www.alesait.com"}',
 '["chatbot", "ai", "whatsapp", "automation", "demo", "support", "harga", "technology"]', TRUE),

-- Healthcare
('template-003', 'Klinik & Healthcare', 'healthcare', 'Template untuk klinik, doktor, dan perkhidmatan kesihatan', '🏥',
 '{"system": "Anda adalah AI assistant untuk klinik kesihatan. Anda membantu pesakit dengan: Appointment booking, Maklumat perkhidmatan, Waktu operasi, Prosedur dan harga, Emergency contact. Selalu professional, empati, dan prioritas emergency cases.", "company_info": "🏥 **[NAMA KLINIK]**\\n👨‍⚕️ Doktor: [NAMA DOKTOR]\\n📍 Alamat: [ALAMAT KLINIK]\\n📞 Telefon: [NOMBOR TELEFON]\\n🚨 Emergency: [NOMBOR EMERGENCY]"}',
 '["appointment", "emergency", "harga", "vaksin", "insurance", "doktor", "klinik"]', TRUE),

-- Retail
('template-004', 'Kedai Runcit', 'retail', 'Template untuk kedai runcit, mini market, dan grocery', '🏪',
 '{"system": "Anda adalah AI assistant untuk kedai runcit. Anda membantu pelanggan dengan: Maklumat produk dan stock, Harga barang-barang, Promosi dan diskaun, Delivery service, Waktu operasi. Selalu mesra, helpful, dan promosi deals yang menarik.", "company_info": "🏪 **[NAMA KEDAI RUNCIT]**\\n📍 Alamat: [ALAMAT KEDAI]\\n📞 Telefon: [NOMBOR TELEFON]\\n⏰ Waktu: [WAKTU OPERASI]\\n🚗 Delivery: [YA/TIDAK]"}',
 '["beras", "minyak", "susu", "delivery", "promo", "grocery", "kedai"]', TRUE),

-- Beauty
('template-005', 'Salon Kecantikan', 'beauty', 'Template untuk salon, spa, dan perkhidmatan kecantikan', '💄',
 '{"system": "Anda adalah AI assistant untuk salon kecantikan. Anda membantu pelanggan dengan: Booking appointment, Maklumat treatment dan harga, Package dan promosi, Beauty consultation, Produk dan skincare. Selalu mesra, inspiring, dan focus pada beauty goals pelanggan.", "company_info": "💄 **[NAMA SALON]**\\n✨ Speciality: [JENIS TREATMENT]\\n📍 Alamat: [ALAMAT SALON]\\n📞 Telefon: [NOMBOR TELEFON]\\n⏰ Waktu: [WAKTU OPERASI]"}',
 '["facial", "hair", "manicure", "package", "booking", "beauty", "salon"]', TRUE),

-- Technology Support
('template-006', 'Kedai Komputer & IT', 'technology', 'Template untuk kedai komputer, repair laptop, dan IT services', '💻',
 '{"system": "Anda adalah AI assistant untuk kedai komputer dan IT services. Anda membantu pelanggan dengan: Repair laptop dan komputer, Jual beli hardware, Software installation, Data recovery, IT consultation. Selalu technical tapi mudah faham, professional, dan solve problem customer.", "company_info": "💻 **[NAMA KEDAI IT]**\\n🔧 Speciality: Computer Repair\\n📍 Alamat: [ALAMAT]\\n📞 Telefon: [NOMBOR TELEFON]"}',
 '["repair", "laptop", "computer", "data", "hardware", "software", "it"]', TRUE),

-- Sports
('template-007', 'Kedai Basikal', 'sports', 'Template untuk kedai basikal, repair, dan aksesoris', '🚴',
 '{"system": "Anda adalah AI assistant untuk kedai basikal. Anda membantu pelanggan dengan: Jual beli basikal, Repair dan maintenance, Aksesoris dan parts, Consultation pembelian, Custom modification. Selalu enthusiastic tentang cycling, share tips, dan help customer pilih yang sesuai.", "company_info": "🚴 **[NAMA KEDAI BASIKAL]**\\n🏔️ Speciality: Mountain & Road Bikes\\n📍 Alamat: [ALAMAT]\\n📞 Telefon: [NOMBOR TELEFON]"}',
 '["basikal", "repair", "mountain", "road", "cycling", "accessories", "bike"]', TRUE),

-- Hospitality
('template-008', 'Homestay & Penginapan', 'hospitality', 'Template untuk homestay, hotel budget, dan accommodation', '🏠',
 '{"system": "Anda adalah AI assistant untuk homestay. Anda membantu tetamu dengan: Room booking dan availability, Harga dan package, Facilities dan amenities, Check-in/out procedure, Local attractions info. Selalu welcoming, helpful, dan make guests feel at home.", "company_info": "🏠 **[NAMA HOMESTAY]**\\n🏨 Comfortable Stay\\n📍 Alamat: [ALAMAT]\\n📞 Telefon: [NOMBOR TELEFON]\\n⏰ Check-in: [WAKTU]"}',
 '["booking", "facilities", "room", "homestay", "accommodation", "tourism"]', TRUE),

-- Jewelry
('template-009', 'Kedai Emas & Jewelry', 'jewelry', 'Template untuk kedai emas, jewelry, dan aksesoris', '💍',
 '{"system": "Anda adalah AI assistant untuk kedai emas. Anda membantu pelanggan dengan: Harga emas terkini, Jenis jewelry available, Custom design, Trade-in service, Investment consultation. Selalu professional, trustworthy, dan share market updates.", "company_info": "💍 **[NAMA KEDAI EMAS]**\\n🥇 Gold & Jewelry\\n📍 Alamat: [ALAMAT]\\n📞 Telefon: [NOMBOR TELEFON]\\n💰 Trade-in Available"}',
 '["harga", "emas", "cincin", "jewelry", "gold", "investment", "custom"]', TRUE),

-- Flowers
('template-010', 'Kedai Bunga', 'flowers', 'Template untuk kedai bunga, florist, dan decoration', '🌸',
 '{"system": "Anda adalah AI assistant untuk kedai bunga. Anda membantu pelanggan dengan: Fresh flowers dan arrangement, Wedding decoration, Funeral wreath, Special occasion bouquet, Delivery service. Selalu creative, romantic, dan help customer express emotions through flowers.", "company_info": "🌸 **[NAMA KEDAI BUNGA]**\\n🌹 Fresh Flowers Daily\\n📍 Alamat: [ALAMAT]\\n📞 Telefon: [NOMBOR TELEFON]\\n🚗 Delivery Available"}',
 '["buket", "wedding", "funeral", "flowers", "decoration", "romantic", "fresh"]', TRUE),

-- Automotive
('template-011', 'Bengkel Kereta', 'automotive', 'Template untuk bengkel kereta, workshop, dan car services', '🚗',
 '{"system": "Anda adalah AI assistant untuk bengkel kereta. Anda membantu pelanggan dengan: Car servicing dan repair, Spare parts, Car inspection, Emergency roadside, Insurance claims. Selalu reliable, trustworthy, dan prioritas safety.", "company_info": "🚗 **[NAMA BENGKEL]**\\n🔧 Car Service Expert\\n📍 Alamat: [ALAMAT]\\n📞 Telefon: [NOMBOR TELEFON]\\n🚨 Emergency: [NOMBOR]"}',
 '["service", "repair", "spare", "inspection", "emergency", "insurance", "car"]', TRUE),

-- Fashion
('template-012', 'Butik Pakaian', 'fashion', 'Template untuk butik, fashion, dan pakaian', '👗',
 '{"system": "Anda adalah AI assistant untuk butik pakaian. Anda membantu pelanggan dengan: Latest fashion trends, Size consultation, Custom tailoring, Wedding outfits, Accessories matching. Selalu stylish, trendy, dan help customer look their best.", "company_info": "👗 **[NAMA BUTIK]**\\n✨ Fashion & Style\\n📍 Alamat: [ALAMAT]\\n📞 Telefon: [NOMBOR TELEFON]\\n👨‍🎨 Custom Tailoring"}',
 '["baju", "fashion", "trend", "custom", "wedding", "accessories", "style"]', TRUE),

-- Education
('template-013', 'Pusat Tuisyen', 'education', 'Template untuk pusat tuisyen, tutor, dan pendidikan', '📚',
 '{"system": "Anda adalah AI assistant untuk pusat tuisyen. Anda membantu ibu bapa dan pelajar dengan: Subject tutoring, Exam preparation, Class schedules, Fee information, Teacher qualifications. Selalu encouraging, academic focused, dan motivate students.", "company_info": "📚 **[NAMA PUSAT TUISYEN]**\\n🎓 Academic Excellence\\n📍 Alamat: [ALAMAT]\\n📞 Telefon: [NOMBOR TELEFON]\\n⏰ Kelas: [WAKTU]"}',
 '["tuisyen", "exam", "math", "science", "english", "spm", "upsr", "pt3"]', TRUE),

-- Pet Services
('template-014', 'Pet Shop & Veterinar', 'pets', 'Template untuk pet shop, vet clinic, dan pet services', '🐾',
 '{"system": "Anda adalah AI assistant untuk pet shop dan veterinar. Anda membantu pet owners dengan: Pet health consultation, Pet food dan accessories, Grooming services, Emergency pet care, Pet boarding. Selalu caring, animal loving, dan prioritas pet welfare.", "company_info": "🐾 **[NAMA PET SHOP]**\\n🐕 Pet Care Expert\\n📍 Alamat: [ALAMAT]\\n📞 Telefon: [NOMBOR TELEFON]\\n🚨 Emergency: [NOMBOR]"}',
 '["pet", "grooming", "food", "vet", "vaccination", "boarding", "emergency"]', TRUE),

-- Financial Services
('template-015', 'Ejen Insurance', 'finance', 'Template untuk ejen insurance, takaful, dan financial planning', '💼',
 '{"system": "Anda adalah AI assistant untuk ejen insurance. Anda membantu pelanggan dengan: Insurance quotation, Policy comparison, Claims assistance, Financial planning, Investment advice. Selalu professional, trustworthy, dan explain complex terms simply.", "company_info": "💼 **[NAMA EJEN]**\\n🛡️ Insurance & Financial Planning\\n📍 Kawasan: [KAWASAN COVERAGE]\\n📞 Telefon: [NOMBOR TELEFON]\\n💬 WhatsApp: [NOMBOR]"}',
 '["insurance", "claim", "premium", "investment", "takaful", "medical", "motor"]', TRUE);

-- Global commands data (Comprehensive anti-hallucination commands)
INSERT IGNORE INTO global_commands (id, command, response, category, priority, is_active) VALUES

-- System Identity
('global-001', 'AI', 'Saya chatbot untuk survey bisnes asnaf. Fokus saya adalah membantu anda dengan soalan tentang peluang bisnes.', 'system', 1, TRUE),
('global-002', 'siapa awak', 'Saya assistant AI untuk survey bisnes asnaf. Saya nak tahu tentang minat bisnes anda.', 'identity', 2, TRUE),
('global-003', 'apa kerja awak', 'Kerja saya adalah mengumpul maklumat tentang potensi bisnes dalam komuniti asnaf melalui survey.', 'identity', 2, TRUE),
('global-004', 'kenapa awak tanya', 'Survey ini untuk membantu kerajaan memahami potensi bisnes dalam komuniti asnaf dan merancang bantuan yang sesuai.', 'explanation', 2, TRUE),

-- Business Focus Commands
('global-005', 'bisnes apa', 'Apa jenis bisnes yang anda berminat? Contoh: kedai makan, online shop, jahit, bengkel, atau lain-lain?', 'business', 1, TRUE),
('global-006', 'modal', 'Berapa modal yang anda ada untuk memulakan bisnes? RM1,000? RM5,000? Atau lebih?', 'business', 1, TRUE),
('global-007', 'skill', 'Apa kemahiran atau pengalaman yang anda ada untuk bisnes ini?', 'business', 1, TRUE),
('global-008', 'lokasi', 'Kat mana anda rancang nak buka bisnes ni? Rumah? Sewa kedai? Atau online?', 'business', 1, TRUE),

-- Restrictions
('global-009', 'politik', 'Maaf, saya fokus pada survey bisnes sahaja. Mari kita berbual tentang peluang bisnes anda.', 'restriction', 3, TRUE),
('global-010', 'agama', 'Saya tidak boleh bincang topik agama. Mari fokus pada survey bisnes dan peluang ekonomi.', 'restriction', 3, TRUE),
('global-011', 'gossip', 'Saya cuma fokus pada survey bisnes. Boleh cerita tentang idea bisnes anda?', 'restriction', 3, TRUE),
('global-012', 'personal', 'Mari kita fokus pada survey bisnes. Apa jenis bisnes yang anda berminat?', 'restriction', 3, TRUE),

-- Encouragement
('global-013', 'tak tahu', 'Tak apa, kita boleh explore sama-sama. Apa yang anda suka buat? Masak? Craft? Atau ada hobby lain?', 'encouragement', 2, TRUE),
('global-014', 'takut', 'Risau memang normal. Kita boleh mulakan kecil-kecilan dulu. Apa yang paling mudah untuk anda buat?', 'encouragement', 2, TRUE),
('global-015', 'susah', 'Setiap bisnes ada cabaran, tapi dengan planning yang betul boleh berjaya. Nak cuba apa dulu?', 'encouragement', 2, TRUE),

-- Common Malaysian Responses
('global-016', 'ok', 'Bagus! Boleh cerita lebih detail tentang idea bisnes anda?', 'response', 1, TRUE),
('global-017', 'boleh', 'Great! Mari kita discuss lebih lanjut. Apa step pertama yang anda nak ambil?', 'response', 1, TRUE),
('global-018', 'nak cuba', 'Excellent! Saya akan bantu anda plan. Mula-mula, apa jenis bisnes yang menarik minat anda?', 'response', 1, TRUE),

-- Emergency Redirects
('global-019', 'emergency', 'Kalau ada emergency, sila hubungi 999. Saya cuma boleh bantu dengan survey bisnes.', 'emergency', 5, TRUE),
('global-020', 'tolong', 'Saya nak tolong anda dengan survey bisnes. Apa masalah atau soalan tentang bisnes yang anda ada?', 'help', 2, TRUE);

-- Keyword responses data (Comprehensive responses for business templates)
INSERT IGNORE INTO keyword_responses (id, template_id, keyword, response, category, language, is_active) VALUES

-- Restoran & Makanan Keywords
('kw-001', 'template-001', 'menu', '🍽️ **MENU POPULAR KAMI:**\n\n🍛 Nasi Ayam - RM12\n🍜 Mee Goreng - RM8\n🥤 Teh Ais - RM3\n\nNak tengok full menu? Boleh WhatsApp kami!', 'product', 'ms', TRUE),
('kw-002', 'template-001', 'harga', '💰 **SENARAI HARGA:**\n\n🍛 Makanan utama: RM8-15\n🥤 Minuman: RM3-6\n🍰 Dessert: RM5-8\n\n*Harga termasuk cukai*', 'pricing', 'ms', TRUE),
('kw-003', 'template-001', 'delivery', '🚗 **DELIVERY SERVICE:**\n\n✅ Kawasan delivery: 10km radius\n💵 Caj delivery: RM3-5\n⏰ Masa delivery: 30-45 minit\n📱 Order: WhatsApp atau Grab/FoodPanda', 'delivery', 'ms', TRUE),
('kw-004', 'template-001', 'booking', '📅 **TEMPAHAN MEJA:**\n\nBoleh book meja untuk:\n🕐 Lunch: 12pm-3pm\n🕕 Dinner: 6pm-10pm\n\nWhatsApp nama, tarikh, masa & berapa orang ya! 😊', 'support', 'ms', TRUE),

-- Alesa IT Services Keywords  
('kw-005', 'template-002', 'chatbot', '🤖 **AI CHATBOT SOLUTIONS:**\n\n✨ Custom AI Chatbot: RM5,000-15,000\n📱 WhatsApp Integration: RM3,000-8,000\n🎯 Lead Generation Bot: RM4,000-10,000\n📊 Analytics Dashboard: RM2,000-5,000\n\n✅ 30 hari trial period', 'product', 'ms', TRUE),
('kw-006', 'template-002', 'ai', '🧠 **AI SOLUTIONS:**\n\n🤖 Natural Language Processing\n📊 Machine Learning Models\n🎯 Predictive Analytics\n💬 Conversational AI\n📈 Business Intelligence\n\n🏆 Malaysia Leading AI Company', 'product', 'ms', TRUE),
('kw-007', 'template-002', 'whatsapp', '📱 **WHATSAPP BUSINESS API:**\n\n🤖 AI Auto Reply: RM2,500-6,000\n📊 CRM Integration: RM4,000-10,000\n🎯 Marketing Automation: RM3,000-8,000\n📈 Analytics & Reports: RM1,500-4,000', 'product', 'ms', TRUE),
('kw-008', 'template-002', 'demo', '🎯 **FREE DEMO & CONSULTATION:**\n\n✅ Live chatbot demonstration\n📊 ROI analysis untuk bisnes anda\n🎯 Custom solution recommendation\n💼 Implementation timeline', 'support', 'ms', TRUE),

-- Healthcare Keywords
('kw-009', 'template-003', 'appointment', '📅 **TEMUJANJI KLINIK:**\n\n⏰ Waktu available:\n🌅 Pagi: 9am-12pm\n🌇 Petang: 2pm-6pm\n\nWhatsApp nama, IC & preferred time ya! 😊', 'support', 'ms', TRUE),
('kw-010', 'template-003', 'emergency', '🚨 **EMERGENCY CONTACT:**\n\n📞 Hotline: [NOMBOR EMERGENCY]\n🏥 24/7 emergency care\n🚑 Serious cases → Hospital\n\n⚠️ Call immediately untuk emergency!', 'support', 'ms', TRUE),
('kw-011', 'template-003', 'vaksin', '💉 **PERKHIDMATAN VAKSIN:**\n\n✅ COVID-19 booster\n✅ Flu vaccine\n✅ Hepatitis B\n✅ Travel vaccination\n\nCall untuk appointment & stock availability!', 'product', 'ms', TRUE),

-- Kedai Runcit Keywords
('kw-012', 'template-004', 'beras', '🍚 **BERAS QUALITY:**\n\n🌾 Beras Thailand AAA - RM65/10kg\n🌾 Beras tempatan - RM45/10kg\n🌾 Beras organik - RM85/10kg\n\n✅ Fresh stock mingguan!', 'product', 'ms', TRUE),
('kw-013', 'template-004', 'minyak', '🫒 **MINYAK MASAK:**\n\n🥥 Minyak kelapa - RM18/1kg\n🌻 Minyak bunga matahari - RM12/1kg\n🫒 Minyak zaitun - RM25/500ml\n\n📦 Borong murah available!', 'product', 'ms', TRUE),
('kw-014', 'template-004', 'promo', '🎉 **PROMOSI TERKINI:**\n\n🔥 Buy 2 Free 1 selected items\n💰 Member discount 5-10%\n📦 Bulk order discount\n🎁 Weekly special deals', 'general', 'ms', TRUE),

-- Beauty Salon Keywords
('kw-015', 'template-005', 'facial', '✨ **FACIAL TREATMENT:**\n\n🌸 Basic Facial - RM80\n💎 Diamond Peel - RM150\n🧴 Hydrating Facial - RM120\n⚡ Acne Treatment - RM100\n\n📅 Book appointment sekarang!', 'product', 'ms', TRUE),
('kw-016', 'template-005', 'hair', '💇‍♀️ **HAIR SERVICES:**\n\n✂️ Hair Cut - RM25-45\n🎨 Hair Color - RM80-150\n🌊 Rebonding - RM180-250\n💫 Hair Treatment - RM60-100', 'product', 'ms', TRUE),
('kw-017', 'template-005', 'package', '📦 **BEAUTY PACKAGES:**\n\n👸 Bridal Package - RM350++\n💄 Party Makeup - RM120\n✨ Monthly Package - RM200\n🎁 Student discount 20%', 'general', 'ms', TRUE),

-- IT Repair Keywords
('kw-018', 'template-006', 'repair', '🔧 **COMPUTER REPAIR:**\n\n💻 Laptop repair: RM50-200\n🖥️ Desktop repair: RM40-150\n📱 Phone repair: RM30-180\n💾 Data recovery: RM80-300\n\n⚡ Same day service!', 'product', 'ms', TRUE),
('kw-019', 'template-006', 'laptop', '💻 **LAPTOP SERVICES:**\n\n🔧 Screen replacement: RM150-400\n⌨️ Keyboard repair: RM80-120\n🔋 Battery replacement: RM100-200\n❄️ Cooling system: RM60-100', 'product', 'ms', TRUE),

-- Bicycle Shop Keywords
('kw-020', 'template-007', 'basikal', '🚴 **JENIS BASIKAL:**\n\n🏔️ Mountain bike: RM300-2000\n🏃 Road bike: RM500-3000\n🚲 Hybrid bike: RM250-800\n⚡ Electric bike: RM1500-5000\n\nTest ride available!', 'product', 'ms', TRUE);

-- Create sample users first (needed for conversations foreign key)
INSERT IGNORE INTO users (id, tenant_id, phone_number, chat_id, name, profile_name, language, is_active) VALUES
('user-001', 'a1ffbb8c-cb66-43de-8d04-175289124254', '60123456789', 'chat-001', 'Siti Aminah', 'Siti', 'ms', TRUE),
('user-002', 'a1ffbb8c-cb66-43de-8d04-175289124254', '60198765432', 'chat-002', 'Ahmad Rahman', 'Ahmad', 'ms', TRUE),
('user-003', 'a1ffbb8c-cb66-43de-8d04-175289124254', '60187654321', 'chat-003', 'Fatimah Zahra', 'Fatimah', 'ms', TRUE);

-- Sample conversations data (Real conversation examples)
INSERT IGNORE INTO conversations (id, user_id, tenant_id, survey_id, title, status, language) VALUES
('conv-001', 'user-001', 'a1ffbb8c-cb66-43de-8d04-175289124254', 'survey-001', 'Business Survey - Siti Aminah', 'archived', 'ms'),
('conv-002', 'user-002', 'a1ffbb8c-cb66-43de-8d04-175289124254', 'survey-001', 'Business Survey - Ahmad Rahman', 'active', 'ms'),
('conv-003', 'user-003', 'a1ffbb8c-cb66-43de-8d04-175289124254', 'survey-001', 'Business Survey - Fatimah Zahra', 'archived', 'ms');

-- Sample messages for conversations
INSERT IGNORE INTO messages (id, conversation_id, content, sender_type, message_type) VALUES
('msg-001', 'conv-001', 'Assalamualaikum! Saya dari survey bisnes asnaf. Boleh tanya, ada minat nak buka bisnes tak?', 'bot', 'text'),
('msg-002', 'conv-001', 'Waalaikumussalam. Ada, saya nak buka kedai kuih.', 'user', 'text'),
('msg-003', 'conv-001', 'Bagus! Kuih apa yang anda pandai buat?', 'bot', 'text'),
('msg-004', 'conv-001', 'Saya pandai buat kuih raya, pau, karipap semua.', 'user', 'text'),
('msg-005', 'conv-002', 'Hi! Survey bisnes asnaf ni. Anda ada minat nak venture into any business?', 'bot', 'text'),
('msg-006', 'conv-002', 'Ada, tapi tak sure nak buat apa. Maybe online je.', 'user', 'text'),
('msg-007', 'conv-002', 'Online business bagus! Apa skill atau hobi yang anda ada?', 'bot', 'text'),
('msg-008', 'conv-003', 'Selamat pagi! Survey untuk asnaf community tentang business potential. Ada plan nak start bisnes?', 'bot', 'text'),
('msg-009', 'conv-003', 'Selamat pagi. Saya nak buka tailoring services dari rumah.', 'user', 'text'),
('msg-010', 'conv-003', 'Excellent! Berapa lama pengalaman jahit anda?', 'bot', 'text'),
('msg-011', 'conv-003', '15 tahun dah. Biasa jahit baju kurung, kebaya semua.', 'user', 'text');

-- Sample demographic data
INSERT IGNORE INTO demographic_data (id, user_id, tenant_id, age_range, gender, race, education_level, occupation, income_range, location, family_size, confidence_score) VALUES

('demo-001', 'user-001', 'a1ffbb8c-cb66-43de-8d04-175289124254', '30-39', 'female', 'malay', 'spm', 'housewife', 'below_1000', 'shah alam', 5, 0.85),

('demo-002', 'user-002', 'a1ffbb8c-cb66-43de-8d04-175289124254', '25-29', 'male', 'malay', 'diploma', 'clerk', '1000_3000', 'kuala lumpur', 1, 0.78),

('demo-003', 'user-003', 'a1ffbb8c-cb66-43de-8d04-175289124254', '40-49', 'female', 'malay', 'spm', 'tailor', '1000_3000', 'johor bahru', 4, 0.92);

-- Sample surveys data (Business assessment surveys)
INSERT IGNORE INTO surveys (id, tenant_id, title, description, is_active) VALUES

('survey-001', 'a1ffbb8c-cb66-43de-8d04-175289124254', 'Business Potential Assessment', 'Comprehensive survey untuk assess business potential dalam asnaf community', TRUE),
('survey-002', 'a1ffbb8c-cb66-43de-8d04-175289124254', 'Asnaf Economic Survey', 'Detailed economic assessment untuk asnaf community development programs', TRUE),
('survey-003', 'a1ffbb8c-cb66-43de-8d04-175289124254', 'Business Readiness Check', 'Quick assessment untuk determine business readiness level among asnaf participants', TRUE);

-- =============================================
-- PERFORMANCE OPTIMIZATIONS
-- =============================================

-- Additional indexes for better performance
CREATE INDEX IF NOT EXISTS idx_users_tenant_active ON users(tenant_id, is_active, last_activity);
CREATE INDEX IF NOT EXISTS idx_conversations_tenant_last_message ON conversations(tenant_id, last_message_at DESC);
CREATE INDEX IF NOT EXISTS idx_messages_conversation_created ON messages(conversation_id, created_at DESC);
CREATE INDEX IF NOT EXISTS idx_survey_responses_tenant_analyzed ON survey_responses(tenant_id, analyzed_at DESC);
CREATE INDEX IF NOT EXISTS idx_demographics_confidence ON demographic_data(confidence_score DESC);
CREATE INDEX IF NOT EXISTS idx_customer_leads_tenant_score ON customer_leads(tenant_id, lead_score DESC);

-- =============================================
-- DATABASE SETUP COMPLETE
-- =============================================