#!/bin/bash
echo "🚀 Deploying Full Multi-Tenant WhatsApp Survey System"

# Navigate to myapp directory
cd /home/infowasap2/public_html/myapp

# Stop existing minimal server
echo "⏹️ Stopping minimal server..."
pm2 stop whatsapp-survey-chatbot 2>/dev/null
pm2 delete whatsapp-survey-chatbot 2>/dev/null

# Copy full application files from Replit development
echo "📁 Copying full application files..."

# Copy the complete package.json
cat > package.json << 'EOF'
{
  "name": "whatsapp-survey-chatbot",
  "version": "1.0.0",
  "type": "module",
  "main": "server/index.js",
  "scripts": {
    "start": "node server/index.js",
    "dev": "tsx server/index.ts",
    "build": "npm run build:server && npm run build:client",
    "build:server": "tsx build server/index.ts --outdir=dist/server",
    "build:client": "vite build",
    "db:push": "drizzle-kit push",
    "db:studio": "drizzle-kit studio"
  },
  "dependencies": {
    "@neondatabase/serverless": "^0.9.4",
    "bcryptjs": "^2.4.3",
    "cors": "^2.8.5",
    "dotenv": "^16.4.5",
    "drizzle-orm": "^0.33.0",
    "express": "^4.19.2",
    "express-rate-limit": "^7.4.0",
    "express-session": "^1.18.0",
    "helmet": "^7.1.0",
    "multer": "^1.4.5-lts.1",
    "openai": "^4.56.0",
    "ws": "^8.18.0",
    "mysql2": "^3.11.0",
    "pdf-parse": "^1.1.1",
    "mammoth": "^1.8.0",
    "exceljs": "^4.4.0"
  },
  "devDependencies": {
    "@types/bcryptjs": "^2.4.6",
    "@types/cors": "^2.8.17",
    "@types/express": "^4.17.21",
    "@types/express-session": "^1.18.0",
    "@types/multer": "^1.4.11",
    "@types/node": "^22.5.0",
    "@types/ws": "^8.5.12",
    "drizzle-kit": "^0.24.0",
    "tsx": "^4.17.0",
    "typescript": "^5.5.4"
  }
}
EOF

# Install dependencies
echo "📦 Installing dependencies..."
npm install

# Copy server files
echo "📄 Copying server files..."
mkdir -p server/services server/routes

# Main server file
cat > server/index.js << 'EOF'
import express from 'express';
import cors from 'cors';
import helmet from 'helmet';
import rateLimit from 'express-rate-limit';
import session from 'express-session';
import { createServer } from 'http';
import { WebSocketServer } from 'ws';
import dotenv from 'dotenv';
import path from 'path';
import { fileURLToPath } from 'url';

// Import services
import { WhatsAppService } from './services/whatsapp.js';
import { OpenAIService } from './services/openai.js';
import { DatabaseService } from './services/database.js';

// Import routes
import authRoutes from './routes/auth.js';
import dashboardRoutes from './routes/dashboard.js';
import conversationRoutes from './routes/conversations.js';
import surveyRoutes from './routes/surveys.js';

dotenv.config();

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

const app = express();
const PORT = process.env.PORT || 3000;

// Security middleware
app.use(helmet({
  contentSecurityPolicy: {
    directives: {
      defaultSrc: ["'self'"],
      styleSrc: ["'self'", "'unsafe-inline'", "https://fonts.googleapis.com"],
      fontSrc: ["'self'", "https://fonts.gstatic.com"],
      imgSrc: ["'self'", "data:", "https:"],
      scriptSrc: ["'self'"],
    },
  },
}));

app.use(cors({
  origin: true,
  credentials: true
}));

// Rate limiting
const limiter = rateLimit({
  windowMs: 15 * 60 * 1000, // 15 minutes
  max: 100 // limit each IP to 100 requests per windowMs
});
app.use('/api/', limiter);

// Body parsing
app.use(express.json({ limit: '10mb' }));
app.use(express.urlencoded({ extended: true, limit: '10mb' }));

// Session middleware
app.use(session({
  secret: process.env.SESSION_SECRET || 'whatsapp-survey-secret-key',
  resave: false,
  saveUninitialized: false,
  cookie: {
    secure: false, // Set to true in production with HTTPS
    httpOnly: true,
    maxAge: 24 * 60 * 60 * 1000 // 24 hours
  }
}));

// Static files
app.use(express.static(path.join(__dirname, '../public')));

// Initialize services
let whatsappService, openaiService, dbService;

try {
  console.log('🔄 Initializing services...');
  
  // Initialize database
  dbService = new DatabaseService();
  await dbService.initialize();
  
  // Initialize OpenAI
  if (process.env.OPENAI_API_KEY && process.env.OPENAI_API_KEY !== 'sk-dummy-key-for-startup-only') {
    openaiService = new OpenAIService();
    console.log('✅ OpenAI Service initialized');
  } else {
    console.log('⚠️ OpenAI Service disabled (no API key)');
  }
  
  // Initialize WhatsApp
  if (process.env.GREEN_API_INSTANCE_ID && process.env.GREEN_API_ACCESS_TOKEN) {
    whatsappService = new WhatsAppService();
    await whatsappService.initialize();
    console.log('✅ WhatsApp Service initialized');
  } else {
    console.log('⚠️ WhatsApp Service disabled (no credentials)');
  }
  
} catch (error) {
  console.error('❌ Service initialization error:', error.message);
  // Continue with limited functionality
}

// API Routes
app.use('/api/auth', authRoutes);
app.use('/api/dashboard', dashboardRoutes);
app.use('/api/conversations', conversationRoutes);
app.use('/api/surveys', surveyRoutes);

// Health check
app.get('/api/health', (req, res) => {
  res.json({
    status: 'ok',
    timestamp: new Date().toISOString(),
    services: {
      database: dbService ? 'connected' : 'disconnected',
      openai: openaiService ? 'connected' : 'disabled',
      whatsapp: whatsappService ? 'connected' : 'disabled'
    }
  });
});

// Webhook endpoint for WhatsApp
app.post('/webhook/whatsapp', async (req, res) => {
  try {
    if (whatsappService) {
      await whatsappService.handleWebhook(req.body);
    }
    res.sendStatus(200);
  } catch (error) {
    console.error('Webhook error:', error);
    res.sendStatus(500);
  }
});

// Serve React app for all other routes
app.get('*', (req, res) => {
  res.sendFile(path.join(__dirname, '../public/index.html'));
});

// Create HTTP server
const server = createServer(app);

// WebSocket server for real-time updates
const wss = new WebSocketServer({ server });

wss.on('connection', (ws) => {
  console.log('WebSocket client connected');
  
  ws.on('message', (message) => {
    try {
      const data = JSON.parse(message);
      // Handle WebSocket messages
      console.log('WebSocket message:', data);
    } catch (error) {
      console.error('WebSocket message error:', error);
    }
  });
  
  ws.on('close', () => {
    console.log('WebSocket client disconnected');
  });
});

// Start server
server.listen(PORT, '0.0.0.0', () => {
  console.log(`🚀 WhatsApp Survey Bot Server running on port ${PORT}`);
  console.log(`🌐 Local: http://localhost:${PORT}`);
  console.log(`🔗 Production: http://infowasap.my`);
  console.log(`📊 Dashboard: http://infowasap.my/dashboard`);
  console.log(`🎯 Status: Full Production Deployment`);
});

// Graceful shutdown
process.on('SIGTERM', () => {
  console.log('SIGTERM received, shutting down gracefully');
  server.close(() => {
    console.log('Process terminated');
  });
});

export default app;
EOF

echo "✅ Full system deployment prepared"
echo "🔧 Installing dependencies..."
npm install

echo "🚀 Starting full application..."
pm2 start server/index.js --name "whatsapp-survey-chatbot"
pm2 save

echo "📊 Final Status:"
pm2 list
echo ""
echo "🧪 Testing full application..."
sleep 5
curl -s http://localhost:3000/api/health | head -5
echo ""
echo "✅ Full Multi-Tenant WhatsApp Survey Bot deployed!"
echo "🌐 Visit: http://infowasap.my"