// Fixed OpenAI Service for production - handles missing API keys gracefully
import OpenAI from "openai";
import { whatsappInteractiveService } from "./whatsapp-interactive.js";

export class OpenAIService {
  constructor() {
    const apiKey = process.env.OPENAI_API_KEY || process.env.OPENAI_API_KEY_ENV_VAR;
    if (apiKey) {
      this.openaiClient = new OpenAI({ apiKey });
      console.log("✅ OpenAI service initialized with API key");
    } else {
      console.warn("⚠️ OpenAI API key not found. AI features will be disabled.");
      this.openaiClient = null;
    }
  }

  checkOpenAI() {
    if (!this.openaiClient) {
      console.warn("⚠️ OpenAI service unavailable. Please configure OPENAI_API_KEY.");
      return false;
    }
    return true;
  }

  async updateApiKey(apiKey) {
    this.openaiClient = new OpenAI({ apiKey });
    console.log("✅ OpenAI API key updated");
  }

  async detectLanguage(text) {
    if (!this.checkOpenAI()) {
      return "ms"; // Default to Malay when OpenAI unavailable
    }
    
    try {
      const response = await this.openaiClient.chat.completions.create({
        model: "gpt-4o",
        messages: [
          {
            role: "system",
            content: "Detect if the text is in Malay (ms) or English (en). Respond with JSON: { 'language': 'ms' or 'en' }"
          },
          {
            role: "user",
            content: text
          }
        ],
        response_format: { type: "json_object" },
      });

      const result = JSON.parse(response.choices[0].message.content || "{}");
      return result.language === "ms" ? "ms" : "en";
    } catch (error) {
      console.error("Language detection failed:", error);
      return "ms"; // Default to Malay
    }
  }

  async processMessage(userMessage, conversationHistory, chatbotPrompts, chatbotKeywords, documentContext) {
    console.log(`🤖 Processing message: "${userMessage.substring(0, 50)}..."`);
    
    // Basic keyword responses that don't need AI
    const basicResponses = {
      'hi': 'Hello! How can I help you?',
      'hello': 'Hi there! How can I assist you today?',
      'thanks': 'You\'re welcome!',
      'terima kasih': 'Sama-sama!',
      'bye': 'Goodbye! Have a great day!',
      'selamat tinggal': 'Selamat tinggal! Semoga hari anda menyenangkan!'
    };

    // Try basic responses first
    const msgLower = userMessage.toLowerCase().trim();
    if (basicResponses[msgLower]) {
      return basicResponses[msgLower];
    }

    // If OpenAI is not available, return helpful fallback
    if (!this.checkOpenAI()) {
      return "Thank you for your message. Our system is currently updating. Please try again later or contact us directly for assistance.";
    }

    try {
      const response = await this.openaiClient.chat.completions.create({
        model: "gpt-4o",
        messages: [
          {
            role: "system",
            content: "You are a helpful customer service assistant. Respond politely and professionally."
          },
          {
            role: "user",
            content: userMessage
          }
        ],
        max_tokens: 200,
        temperature: 0.7,
      });

      return response.choices[0].message.content || "I understand your message. How can I help you further?";
    } catch (error) {
      console.error("OpenAI processing failed:", error);
      return "Thank you for your message. I'm here to help - could you please rephrase your question?";
    }
  }

  // Stub methods for compatibility
  detectContactInfo() { return null; }
  categorizeBusinessPotential() { return { category: 'unknown', score: 0 }; }
  extractDemographics() { return {}; }
}

// Export singleton instance
export const openaiService = new OpenAIService();