#!/bin/bash

# WhatsApp Survey Chatbot - Complete cPanel Installation Script
# Make this file executable: chmod +x install.sh

echo "🚀 Starting Complete WhatsApp Survey Chatbot Installation..."
echo "🎯 This includes full AI-powered multi-tenant features"

# Check if Node.js is available
if ! command -v node &> /dev/null; then
    echo "❌ Node.js is not installed. Please install Node.js 18 or higher first."
    exit 1
fi

# Check Node.js version
NODE_VERSION=$(node -v | cut -d'v' -f2 | cut -d'.' -f1)
if [ "$NODE_VERSION" -lt 18 ]; then
    echo "❌ Node.js version $NODE_VERSION is too old. Please upgrade to Node.js 18 or higher."
    exit 1
fi

echo "✅ Node.js $(node -v) detected"

# Create necessary directories
echo "📁 Creating directories..."
mkdir -p uploads
mkdir -p logs
mkdir -p dist
mkdir -p backups

# Set proper permissions
chmod 755 uploads
chmod 755 logs
chmod 755 dist

# Copy environment file
if [ ! -f .env ]; then
    echo "📝 Creating environment file..."
    cp .env.example .env
    echo "⚠️  IMPORTANT: Please edit .env file with your actual configuration before starting!"
    echo "    Required: DATABASE_URL, GREEN_API_INSTANCE_ID, GREEN_API_ACCESS_TOKEN, OPENAI_API_KEY"
else
    echo "✅ Environment file already exists"
fi

# Install dependencies
echo "📦 Installing dependencies (this may take a few minutes)..."
npm install --production

if [ $? -ne 0 ]; then
    echo "❌ Failed to install dependencies"
    exit 1
fi

echo "✅ Dependencies installed successfully"

# Create startup script
echo "📝 Creating startup script..."
cat > start.sh << 'EOF'
#!/bin/bash
export NODE_ENV=production
export PORT=3000

# Load environment variables
if [ -f .env ]; then
    export $(cat .env | grep -v '#' | grep -v '^$' | awk '/=/ {print $1}')
fi

# Check if required variables are set
if [ -z "$DATABASE_URL" ]; then
    echo "❌ DATABASE_URL not set in .env file"
    exit 1
fi

if [ -z "$GREEN_API_INSTANCE_ID" ]; then
    echo "❌ GREEN_API_INSTANCE_ID not set in .env file"
    exit 1
fi

# Start the application
echo "🚀 Starting WhatsApp Survey Chatbot Platform..."
echo "📊 Admin Panel: http://localhost:3000/admin"
echo "🔗 Webhook URL: http://yourdomain.com/api/green-api/webhook"
node server/index.js
EOF

chmod +x start.sh

# Create PM2 ecosystem file
echo "📝 Creating PM2 configuration..."
cat > ecosystem.config.js << 'EOF'
module.exports = {
  apps: [{
    name: 'whatsapp-survey-chatbot',
    script: 'server/index.js',
    instances: 1,
    exec_mode: 'fork',
    env: {
      NODE_ENV: 'production',
      PORT: 3000
    },
    log_file: 'logs/combined.log',
    out_file: 'logs/out.log',
    error_file: 'logs/error.log',
    log_date_format: 'YYYY-MM-DD HH:mm:ss Z',
    merge_logs: true,
    max_restarts: 10,
    min_uptime: '10s',
    restart_delay: 4000,
    watch: false,
    ignore_watch: ['node_modules', 'logs', 'uploads']
  }]
};
EOF

# Create database setup script
echo "📝 Creating database setup script..."
cat > setup-database.js << 'EOF'
import { testConnection, initializeDatabase } from './server/db.js';
import { storage } from './server/storage.js';
import dotenv from 'dotenv';

dotenv.config();

async function setupDatabase() {
  try {
    console.log('🔄 Testing database connection...');
    const connected = await testConnection();
    
    if (!connected) {
      console.error('❌ Database connection failed. Please check your DATABASE_URL and credentials.');
      process.exit(1);
    }

    console.log('🔄 Initializing database schema...');
    const initialized = await initializeDatabase();
    
    if (!initialized) {
      console.error('❌ Database schema initialization failed.');
      process.exit(1);
    }

    console.log('🔄 Creating default admin user...');
    const adminUsername = process.env.ADMIN_USERNAME || 'admin';
    const adminPassword = process.env.ADMIN_PASSWORD || '##Master8899';
    
    try {
      const existingAdmin = await storage.getAdminUser(adminUsername);
      if (!existingAdmin) {
        await storage.createAdminUser(adminUsername, adminPassword, 'admin@system.com');
        console.log(`✅ Admin user created: ${adminUsername}`);
      } else {
        console.log(`✅ Admin user already exists: ${adminUsername}`);
      }
    } catch (error) {
      console.warn('⚠️ Admin user creation skipped:', error.message);
    }

    console.log('✅ Database setup completed successfully!');
    console.log('');
    console.log('📋 Next steps:');
    console.log('1. Configure your .env file with proper credentials');
    console.log('2. Start the server: ./start.sh or pm2 start ecosystem.config.js');
    console.log('');
    
  } catch (error) {
    console.error('❌ Database setup failed:', error);
    process.exit(1);
  }
}

setupDatabase();
EOF

# Create backup script
echo "📝 Creating backup script..."
cat > backup.sh << 'EOF'
#!/bin/bash
# Backup script for WhatsApp Survey Chatbot

DATE=$(date +%Y%m%d_%H%M%S)
BACKUP_DIR="backups"

echo "📦 Creating backup: backup_$DATE"

# Database backup
if [ ! -z "$DB_USER" ] && [ ! -z "$DB_PASSWORD" ] && [ ! -z "$DB_NAME" ]; then
    echo "🗄️ Backing up database..."
    mysqldump -u $DB_USER -p$DB_PASSWORD $DB_NAME > $BACKUP_DIR/database_$DATE.sql
    echo "✅ Database backup created"
fi

# Files backup
echo "📁 Backing up files..."
tar -czf $BACKUP_DIR/files_$DATE.tar.gz \
    --exclude='node_modules' \
    --exclude='logs' \
    --exclude='backups' \
    --exclude='.git' \
    .

echo "✅ Files backup created"

# Keep only last 7 backups
find $BACKUP_DIR -name "*.sql" -mtime +7 -delete
find $BACKUP_DIR -name "*.tar.gz" -mtime +7 -delete

echo "🧹 Old backups cleaned up"
echo "📦 Backup completed: $BACKUP_DIR/backup_$DATE"
EOF

chmod +x backup.sh

# Create log rotation script
echo "📝 Creating log rotation script..."
cat > rotate-logs.sh << 'EOF'
#!/bin/bash
# Log rotation for WhatsApp Survey Chatbot

LOG_DIR="logs"
DATE=$(date +%Y%m%d)

# Rotate PM2 logs
if [ -f "$LOG_DIR/combined.log" ]; then
    mv "$LOG_DIR/combined.log" "$LOG_DIR/combined_$DATE.log"
    touch "$LOG_DIR/combined.log"
fi

if [ -f "$LOG_DIR/out.log" ]; then
    mv "$LOG_DIR/out.log" "$LOG_DIR/out_$DATE.log"
    touch "$LOG_DIR/out.log"
fi

if [ -f "$LOG_DIR/error.log" ]; then
    mv "$LOG_DIR/error.log" "$LOG_DIR/error_$DATE.log"
    touch "$LOG_DIR/error.log"
fi

# Compress old logs
find $LOG_DIR -name "*_20*.log" -mtime +1 -exec gzip {} \;

# Remove old compressed logs (keep 30 days)
find $LOG_DIR -name "*.log.gz" -mtime +30 -delete

echo "📋 Logs rotated and old logs cleaned up"
EOF

chmod +x rotate-logs.sh

# Create system status check script
echo "📝 Creating status check script..."
cat > check-status.sh << 'EOF'
#!/bin/bash
# System status check for WhatsApp Survey Chatbot

echo "🔍 WhatsApp Survey Chatbot - System Status"
echo "========================================"

# Check if PM2 process is running
if command -v pm2 &> /dev/null; then
    echo "📊 PM2 Status:"
    pm2 status whatsapp-survey-chatbot
    echo ""
fi

# Check database connection
echo "🗄️ Database Status:"
if [ -f .env ]; then
    export $(cat .env | grep -v '#' | grep -v '^$' | awk '/=/ {print $1}')
    if [ ! -z "$DB_USER" ] && [ ! -z "$DB_PASSWORD" ] && [ ! -z "$DB_NAME" ]; then
        mysql -u $DB_USER -p$DB_PASSWORD -e "SELECT 1" $DB_NAME > /dev/null 2>&1
        if [ $? -eq 0 ]; then
            echo "✅ Database connection successful"
        else
            echo "❌ Database connection failed"
        fi
    else
        echo "⚠️ Database credentials not configured"
    fi
else
    echo "❌ .env file not found"
fi

# Check API endpoint
echo ""
echo "🌐 API Status:"
curl -s http://localhost:3000/health > /dev/null 2>&1
if [ $? -eq 0 ]; then
    echo "✅ API endpoint responding"
    curl -s http://localhost:3000/health | jq '.' 2>/dev/null || echo "API response received (jq not available for formatting)"
else
    echo "❌ API endpoint not responding"
fi

# Check disk space
echo ""
echo "💾 Disk Usage:"
df -h . | tail -1 | awk '{print "Used: " $3 " / " $2 " (" $5 ")"}'

# Check logs
echo ""
echo "📋 Recent Logs:"
if [ -f logs/error.log ]; then
    echo "⚠️ Recent errors:"
    tail -5 logs/error.log 2>/dev/null || echo "No recent errors"
else
    echo "✅ No error log file found"
fi

echo ""
echo "========================================"
echo "✅ Status check completed"
EOF

chmod +x check-status.sh

echo "✅ Installation complete!"
echo ""
echo "🎉 ==============================================="
echo "🎯 WhatsApp Survey Chatbot - Installation Complete"
echo "🎉 ==============================================="
echo ""
echo "📋 NEXT STEPS:"
echo ""
echo "1. 📝 CONFIGURE ENVIRONMENT:"
echo "   nano .env"
echo "   # Set your database, WhatsApp API, and OpenAI credentials"
echo ""
echo "2. 🗄️ SETUP DATABASE:"
echo "   node setup-database.js"
echo ""
echo "3. 🚀 START SERVER:"
echo "   # Option A: Direct start"
echo "   ./start.sh"
echo ""
echo "   # Option B: PM2 (recommended for production)"
echo "   pm2 start ecosystem.config.js"
echo "   pm2 startup  # Enable auto-start on reboot"
echo "   pm2 save     # Save PM2 configuration"
echo ""
echo "4. 🔧 CONFIGURE WEBHOOK:"
echo "   Set Green API webhook URL to:"
echo "   https://yourdomain.com/api/green-api/webhook"
echo ""
echo "📊 ADMIN ACCESS:"
echo "   URL: http://localhost:3000/admin"
echo "   Username: admin"
echo "   Password: ##Master8899"
echo ""
echo "🔧 MANAGEMENT SCRIPTS:"
echo "   ./check-status.sh    # Check system status"
echo "   ./backup.sh          # Create backup"
echo "   ./rotate-logs.sh     # Rotate log files"
echo ""
echo "🌐 ENDPOINTS:"
echo "   Health: http://localhost:3000/health"
echo "   API: http://localhost:3000/api/system/status"
echo "   Webhook: http://localhost:3000/api/green-api/webhook"
echo ""
echo "📚 FEATURES INCLUDED:"
echo "   ✅ Multi-tenant WhatsApp chatbot"
echo "   ✅ AI-powered conversation processing"
echo "   ✅ Survey intent analysis & categorization"
echo "   ✅ Demographic data extraction"
echo "   ✅ Keyword tracking & analytics"
echo "   ✅ Interactive WhatsApp messages"
echo "   ✅ Admin dashboard with real-time stats"
echo "   ✅ Business templates system"
echo "   ✅ File upload & management"
echo "   ✅ Complete audit logging"
echo "   ✅ Multi-language support (MS/EN)"
echo ""
echo "🎉 ==============================================="
echo "⚠️  REMEMBER TO:"
echo "   • Configure .env with your actual credentials"
echo "   • Setup SSL certificate for webhook security"
echo "   • Configure domain name in .env"
echo "   • Run database setup before starting"
echo "🎉 ==============================================="