import express from 'express';

const router = express.Router();

// Get all conversations
router.get('/', async (req, res) => {
  try {
    const page = parseInt(req.query.page) || 1;
    const limit = parseInt(req.query.limit) || 20;
    const status = req.query.status;

    // Sample conversation data
    let conversations = [
      {
        id: 'conv1',
        customerName: 'Ahmad Hassan',
        customerPhone: '+60123456789',
        lastMessage: 'Saya nak tanya pasal servis aircond untuk rumah saya',
        timestamp: new Date().toISOString(),
        status: 'active',
        businessPotential: 'high',
        messageCount: 8,
        category: 'Home Services'
      },
      {
        id: 'conv2', 
        customerName: 'Siti Aminah',
        customerPhone: '+60134567890',
        lastMessage: 'Boleh saya dapatkan quotation untuk catering business?',
        timestamp: new Date(Date.now() - 300000).toISOString(),
        status: 'pending',
        businessPotential: 'medium',
        messageCount: 5,
        category: 'Food & Beverage'
      },
      {
        id: 'conv3',
        customerName: 'Lim Wei Ming', 
        customerPhone: '+60145678901',
        lastMessage: 'I want to start online business, need advice',
        timestamp: new Date(Date.now() - 600000).toISOString(),
        status: 'completed',
        businessPotential: 'high',
        messageCount: 12,
        category: 'E-commerce'
      },
      {
        id: 'conv4',
        customerName: 'Fatimah Zahra',
        customerPhone: '+60156789012', 
        lastMessage: 'Saya interested dengan food delivery business',
        timestamp: new Date(Date.now() - 900000).toISOString(),
        status: 'active',
        businessPotential: 'medium',
        messageCount: 6,
        category: 'Food & Beverage'
      },
      {
        id: 'conv5',
        customerName: 'Raj Kumar',
        customerPhone: '+60167890123',
        lastMessage: 'Can you help me with digital marketing for my shop?',
        timestamp: new Date(Date.now() - 1200000).toISOString(),
        status: 'pending', 
        businessPotential: 'high',
        messageCount: 9,
        category: 'Digital Marketing'
      }
    ];

    // Filter by status if provided
    if (status) {
      conversations = conversations.filter(conv => conv.status === status);
    }

    // Pagination
    const startIndex = (page - 1) * limit;
    const endIndex = page * limit;
    const paginatedConversations = conversations.slice(startIndex, endIndex);

    res.json({
      conversations: paginatedConversations,
      pagination: {
        page,
        limit,
        total: conversations.length,
        totalPages: Math.ceil(conversations.length / limit)
      }
    });
  } catch (error) {
    console.error('Conversations error:', error);
    res.status(500).json({ error: 'Failed to fetch conversations' });
  }
});

// Get specific conversation with messages
router.get('/:id', async (req, res) => {
  try {
    const conversationId = req.params.id;

    // Sample conversation with messages
    const conversation = {
      id: conversationId,
      customerName: 'Ahmad Hassan',
      customerPhone: '+60123456789',
      status: 'active',
      businessPotential: 'high',
      category: 'Home Services',
      createdAt: new Date(Date.now() - 3600000).toISOString(),
      demographics: {
        age: '35-45',
        location: 'Kuala Lumpur',
        occupation: 'Business Owner'
      },
      messages: [
        {
          id: 'msg1',
          sender: 'customer',
          content: 'Hello, saya nak tanya pasal servis aircond',
          timestamp: new Date(Date.now() - 3600000).toISOString(),
          type: 'text'
        },
        {
          id: 'msg2', 
          sender: 'bot',
          content: 'Hi Ahmad! Terima kasih for contacting us. What kind of aircond service do you need?',
          timestamp: new Date(Date.now() - 3580000).toISOString(),
          type: 'text'
        },
        {
          id: 'msg3',
          sender: 'customer', 
          content: 'Aircond saya tak sejuk, maybe need service or repair',
          timestamp: new Date(Date.now() - 3570000).toISOString(),
          type: 'text'
        },
        {
          id: 'msg4',
          sender: 'bot',
          content: 'I understand. How old is your aircond unit and what brand is it?',
          timestamp: new Date(Date.now() - 3560000).toISOString(),
          type: 'text'
        }
      ]
    };

    res.json(conversation);
  } catch (error) {
    console.error('Conversation detail error:', error);
    res.status(500).json({ error: 'Failed to fetch conversation details' });
  }
});

// Send message in conversation
router.post('/:id/messages', async (req, res) => {
  try {
    const conversationId = req.params.id;
    const { message, type = 'text' } = req.body;

    if (!message) {
      return res.status(400).json({ error: 'Message content is required' });
    }

    // Here you would integrate with WhatsApp service to send the message
    const newMessage = {
      id: `msg_${Date.now()}`,
      sender: 'bot',
      content: message,
      timestamp: new Date().toISOString(),
      type
    };

    res.json(newMessage);
  } catch (error) {
    console.error('Send message error:', error);
    res.status(500).json({ error: 'Failed to send message' });
  }
});

// Update conversation status
router.patch('/:id/status', async (req, res) => {
  try {
    const conversationId = req.params.id;
    const { status } = req.body;

    if (!['active', 'pending', 'completed'].includes(status)) {
      return res.status(400).json({ error: 'Invalid status' });
    }

    // Here you would update the conversation in your database
    res.json({ 
      success: true, 
      conversationId, 
      newStatus: status 
    });
  } catch (error) {
    console.error('Update status error:', error);
    res.status(500).json({ error: 'Failed to update conversation status' });
  }
});

export default router;