import express from 'express';

const router = express.Router();

// Get all surveys
router.get('/', async (req, res) => {
  try {
    const surveys = [
      {
        id: 'survey1',
        name: 'Air Conditioning Service Survey',
        description: 'Survey for assessing aircond service business potential',
        category: 'Home Services',
        active: true,
        responseCount: 45,
        createdAt: '2025-08-15T00:00:00Z',
        questions: [
          {
            id: 'q1',
            type: 'multiple_choice',
            question: 'What type of aircond service do you need?',
            options: ['Installation', 'Repair', 'Maintenance', 'Replacement']
          },
          {
            id: 'q2', 
            type: 'text',
            question: 'How many aircond units do you have?'
          },
          {
            id: 'q3',
            type: 'multiple_choice',
            question: 'What is your budget range?',
            options: ['Below RM500', 'RM500-1000', 'RM1000-2000', 'Above RM2000']
          }
        ]
      },
      {
        id: 'survey2',
        name: 'Food Business Potential Survey',
        description: 'Assessment for food delivery and catering business opportunities',
        category: 'Food & Beverage',
        active: true,
        responseCount: 32,
        createdAt: '2025-08-14T00:00:00Z',
        questions: [
          {
            id: 'q1',
            type: 'multiple_choice',
            question: 'What type of food business interests you?',
            options: ['Food Delivery', 'Catering', 'Restaurant', 'Food Truck']
          },
          {
            id: 'q2',
            type: 'text', 
            question: 'Do you have cooking experience?'
          },
          {
            id: 'q3',
            type: 'multiple_choice',
            question: 'What is your target market?',
            options: ['Office Workers', 'Students', 'Events', 'General Public']
          }
        ]
      },
      {
        id: 'survey3',
        name: 'Online Business Readiness Survey',
        description: 'Evaluate digital business potential and readiness',
        category: 'E-commerce',
        active: true,
        responseCount: 28,
        createdAt: '2025-08-13T00:00:00Z',
        questions: [
          {
            id: 'q1',
            type: 'multiple_choice',
            question: 'What type of online business are you considering?',
            options: ['E-commerce Store', 'Digital Services', 'Online Course', 'Dropshipping']
          },
          {
            id: 'q2',
            type: 'rating',
            question: 'Rate your digital literacy (1-5)',
            min: 1,
            max: 5
          },
          {
            id: 'q3',
            type: 'text',
            question: 'What products or services would you like to sell online?'
          }
        ]
      }
    ];

    res.json(surveys);
  } catch (error) {
    console.error('Surveys error:', error);
    res.status(500).json({ error: 'Failed to fetch surveys' });
  }
});

// Get specific survey
router.get('/:id', async (req, res) => {
  try {
    const surveyId = req.params.id;
    
    // Sample survey data
    const survey = {
      id: surveyId,
      name: 'Air Conditioning Service Survey',
      description: 'Survey for assessing aircond service business potential',
      category: 'Home Services',
      active: true,
      responseCount: 45,
      createdAt: '2025-08-15T00:00:00Z',
      questions: [
        {
          id: 'q1',
          type: 'multiple_choice',
          question: 'What type of aircond service do you need?',
          options: ['Installation', 'Repair', 'Maintenance', 'Replacement']
        },
        {
          id: 'q2',
          type: 'text', 
          question: 'How many aircond units do you have?'
        },
        {
          id: 'q3',
          type: 'multiple_choice',
          question: 'What is your budget range?',
          options: ['Below RM500', 'RM500-1000', 'RM1000-2000', 'Above RM2000']
        }
      ],
      responses: [
        {
          id: 'resp1',
          conversationId: 'conv1',
          customerName: 'Ahmad Hassan',
          responses: {
            q1: 'Repair',
            q2: '2 units',
            q3: 'RM500-1000'
          },
          businessPotential: 'high',
          completedAt: '2025-08-18T10:30:00Z'
        },
        {
          id: 'resp2',
          conversationId: 'conv5',
          customerName: 'Ali Rahman',
          responses: {
            q1: 'Installation', 
            q2: '1 unit',
            q3: 'RM1000-2000'
          },
          businessPotential: 'medium',
          completedAt: '2025-08-18T09:15:00Z'
        }
      ]
    };

    res.json(survey);
  } catch (error) {
    console.error('Survey detail error:', error);
    res.status(500).json({ error: 'Failed to fetch survey details' });
  }
});

// Create new survey
router.post('/', async (req, res) => {
  try {
    const { name, description, category, questions } = req.body;

    if (!name || !questions || questions.length === 0) {
      return res.status(400).json({ error: 'Name and questions are required' });
    }

    const newSurvey = {
      id: `survey_${Date.now()}`,
      name,
      description,
      category,
      questions,
      active: true,
      responseCount: 0,
      createdAt: new Date().toISOString()
    };

    res.status(201).json(newSurvey);
  } catch (error) {
    console.error('Create survey error:', error);
    res.status(500).json({ error: 'Failed to create survey' });
  }
});

// Update survey
router.put('/:id', async (req, res) => {
  try {
    const surveyId = req.params.id;
    const updates = req.body;

    // Here you would update the survey in your database
    const updatedSurvey = {
      id: surveyId,
      ...updates,
      updatedAt: new Date().toISOString()
    };

    res.json(updatedSurvey);
  } catch (error) {
    console.error('Update survey error:', error);
    res.status(500).json({ error: 'Failed to update survey' });
  }
});

// Get survey responses
router.get('/:id/responses', async (req, res) => {
  try {
    const surveyId = req.params.id;

    const responses = [
      {
        id: 'resp1',
        surveyId,
        conversationId: 'conv1',
        customerName: 'Ahmad Hassan',
        customerPhone: '+60123456789',
        responses: {
          q1: 'Repair',
          q2: '2 units',
          q3: 'RM500-1000'
        },
        demographics: {
          age: '35-45',
          location: 'Kuala Lumpur',
          occupation: 'Business Owner'
        },
        businessPotential: 'high',
        completedAt: '2025-08-18T10:30:00Z'
      },
      {
        id: 'resp2',
        surveyId,
        conversationId: 'conv5', 
        customerName: 'Ali Rahman',
        customerPhone: '+60134567890',
        responses: {
          q1: 'Installation',
          q2: '1 unit', 
          q3: 'RM1000-2000'
        },
        demographics: {
          age: '25-35',
          location: 'Selangor',
          occupation: 'Employee'
        },
        businessPotential: 'medium',
        completedAt: '2025-08-18T09:15:00Z'
      }
    ];

    res.json(responses);
  } catch (error) {
    console.error('Survey responses error:', error);
    res.status(500).json({ error: 'Failed to fetch survey responses' });
  }
});

// Export survey responses to Excel
router.get('/:id/export', async (req, res) => {
  try {
    const surveyId = req.params.id;
    
    // Here you would generate an Excel file with survey responses
    // For now, return a success message
    res.json({ 
      message: 'Export initiated',
      downloadUrl: `/api/surveys/${surveyId}/download`
    });
  } catch (error) {
    console.error('Export survey error:', error);
    res.status(500).json({ error: 'Failed to export survey' });
  }
});

export default router;