import express from 'express';
import cors from 'cors';
import helmet from 'helmet';
import session from 'express-session';
import rateLimit from 'express-rate-limit';
import { createServer } from 'http';
import { WebSocketServer } from 'ws';
import path from 'path';
import { fileURLToPath } from 'url';
import dotenv from 'dotenv';
import fs from 'fs';

// Import custom modules
import { testConnection, initializeDatabase } from './db.js';
import { setupRoutes } from './routes.js';
import { whatsappService } from './services/whatsapp.js';
import { pollingService } from './services/polling.js';
import { botService } from './services/bot.js';
import { openaiService } from './services/openai.js';
import { storage } from './storage.js';

// ES module __dirname equivalent
const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

// Load environment variables
dotenv.config();

const app = express();
const PORT = process.env.PORT || 3000;

// Create upload directory if it doesn't exist
const uploadDir = process.env.UPLOAD_DIR || 'uploads';
if (!fs.existsSync(uploadDir)) {
  fs.mkdirSync(uploadDir, { recursive: true });
  console.log(`📁 Created upload directory: ${uploadDir}`);
}

// Create logs directory
const logsDir = 'logs';
if (!fs.existsSync(logsDir)) {
  fs.mkdirSync(logsDir, { recursive: true });
  console.log(`📁 Created logs directory: ${logsDir}`);
}

// Security middleware
app.use(helmet({
  contentSecurityPolicy: {
    directives: {
      defaultSrc: ["'self'"],
      styleSrc: ["'self'", "'unsafe-inline'", "https:"],
      scriptSrc: ["'self'"],
      imgSrc: ["'self'", "data:", "https:"],
      connectSrc: ["'self'", "wss:", "ws:"],
    },
  },
  crossOriginEmbedderPolicy: false
}));

// CORS configuration
app.use(cors({
  origin: process.env.NODE_ENV === 'production' 
    ? [`https://${process.env.DOMAIN_NAME}`, `https://www.${process.env.DOMAIN_NAME}`]
    : true,
  credentials: true
}));

// Rate limiting
const limiter = rateLimit({
  windowMs: parseInt(process.env.RATE_LIMIT_WINDOW) || 15 * 60 * 1000, // 15 minutes
  max: parseInt(process.env.RATE_LIMIT_MAX) || 100,
  message: { error: 'Too many requests, please try again later.' },
  standardHeaders: true,
  legacyHeaders: false,
});

app.use('/api/', limiter);

// Session configuration
app.use(session({
  secret: process.env.SESSION_SECRET || 'fallback-secret-change-in-production',
  resave: false,
  saveUninitialized: false,
  cookie: {
    secure: process.env.NODE_ENV === 'production',
    httpOnly: true,
    maxAge: 24 * 60 * 60 * 1000 // 24 hours
  }
}));

// Body parsing middleware
app.use(express.json({ limit: '10mb' }));
app.use(express.urlencoded({ extended: true, limit: '10mb' }));

// Serve static files (frontend)
app.use(express.static(path.join(__dirname, '../public')));

// Basic health check (before authentication)
app.get('/health', async (req, res) => {
  const dbStatus = await testConnection();
  res.json({
    status: 'ok',
    timestamp: new Date().toISOString(),
    database: dbStatus ? 'connected' : 'disconnected',
    environment: process.env.NODE_ENV,
    uptime: process.uptime()
  });
});

// Initialize services
async function initializeServices() {
  try {
    console.log('🔄 Initializing services...');
    
    // Test database connection
    const dbConnected = await testConnection();
    if (!dbConnected) {
      throw new Error('Database connection failed');
    }

    // Initialize database schema
    console.log('🔄 Initializing database schema...');
    const schemaInitialized = await initializeDatabase();
    if (!schemaInitialized) {
      console.warn('⚠️ Database schema initialization failed, but continuing...');
    }

    // Create default admin user if not exists
    console.log('🔄 Checking admin user...');
    const adminUser = await storage.getAdminUser(process.env.ADMIN_USERNAME || 'admin');
    if (!adminUser && process.env.ADMIN_USERNAME && process.env.ADMIN_PASSWORD) {
      console.log('👤 Creating default admin user...');
      await storage.createAdminUser(
        process.env.ADMIN_USERNAME,
        process.env.ADMIN_PASSWORD,
        'admin@system.com'
      );
      console.log('✅ Default admin user created');
    } else if (adminUser) {
      console.log('✅ Admin user already exists');
    }

    // Initialize WhatsApp service
    console.log('🔄 Initializing WhatsApp service...');
    // WhatsApp service is already initialized in constructor
    console.log('✅ WhatsApp service initialized');

    // Initialize OpenAI service
    console.log('🔄 Initializing OpenAI service...');
    // OpenAI service is already initialized in constructor
    console.log('✅ OpenAI service initialized');

    // Initialize Bot service
    console.log('🔄 Initializing Bot service...');
    // Bot service is already initialized in constructor
    console.log('✅ Bot service initialized');

    // Initialize polling service
    console.log('🔄 Initializing polling service...');
    await pollingService.start();
    console.log('✅ Polling service started');

    console.log('✅ All services initialized successfully');
    return { whatsappService, botService, pollingService, openaiService };
  } catch (error) {
    console.error('❌ Service initialization failed:', error);
    throw error;
  }
}

// Setup routes
async function setupServer() {
  try {
    const services = await initializeServices();
    
    // Setup API routes
    setupRoutes(app);
    console.log('✅ Routes configured successfully');

    // Catch-all handler for SPA
    app.get('*', (req, res) => {
      res.sendFile(path.join(__dirname, '../public/index.html'));
    });

    return services;
  } catch (error) {
    console.error('❌ Server setup failed:', error);
    process.exit(1);
  }
}

// Create HTTP server
const server = createServer(app);

// Setup WebSocket server
const wss = new WebSocketServer({ server });

wss.on('connection', (ws, req) => {
  console.log('📡 WebSocket client connected from', req.socket.remoteAddress);
  
  // Send welcome message
  ws.send(JSON.stringify({
    type: 'connected',
    message: 'WebSocket connected successfully',
    timestamp: new Date().toISOString()
  }));
  
  ws.on('message', (data) => {
    try {
      const message = JSON.parse(data);
      console.log('📡 WebSocket message received:', message);
      
      // Echo back for now
      ws.send(JSON.stringify({
        type: 'echo',
        data: message,
        timestamp: new Date().toISOString()
      }));
    } catch (error) {
      console.error('📡 WebSocket message error:', error);
    }
  });

  ws.on('close', () => {
    console.log('📡 WebSocket client disconnected');
  });

  ws.on('error', (error) => {
    console.error('📡 WebSocket error:', error);
  });
});

// Graceful shutdown
process.on('SIGTERM', async () => {
  console.log('📡 SIGTERM received, shutting down gracefully');
  
  // Stop polling service
  await pollingService.stop();
  
  server.close(() => {
    console.log('📡 Server closed');
    process.exit(0);
  });
});

process.on('SIGINT', async () => {
  console.log('📡 SIGINT received, shutting down gracefully');
  
  // Stop polling service
  await pollingService.stop();
  
  server.close(() => {
    console.log('📡 Server closed');
    process.exit(0);
  });
});

// Unhandled promise rejection handler
process.on('unhandledRejection', (reason, promise) => {
  console.error('❌ Unhandled Rejection at:', promise, 'reason:', reason);
});

// Uncaught exception handler
process.on('uncaughtException', (error) => {
  console.error('❌ Uncaught Exception:', error);
  process.exit(1);
});

// Start server
async function startServer() {
  try {
    await setupServer();
    
    server.listen(PORT, () => {
      console.log('🎉 ===============================================');
      console.log('🚀 WhatsApp Survey Chatbot Platform Started');
      console.log('🎉 ===============================================');
      console.log(`🌐 Server: http://localhost:${PORT}`);
      console.log(`📊 Admin: http://localhost:${PORT}/admin`);
      console.log(`🔗 Webhook: http://localhost:${PORT}/api/green-api/webhook`);
      console.log(`🌍 Environment: ${process.env.NODE_ENV}`);
      console.log(`📍 Domain: ${process.env.DOMAIN_NAME || 'localhost'}`);
      console.log(`⏰ Started: ${new Date().toISOString()}`);
      console.log('🎉 ===============================================');
      
      // Log system information
      console.log('📋 System Configuration:');
      console.log(`   • WhatsApp Instance: ${process.env.GREEN_API_INSTANCE_ID || 'Not configured'}`);
      console.log(`   • OpenAI: ${process.env.OPENAI_API_KEY ? 'Configured' : 'Not configured'}`);
      console.log(`   • Database: ${process.env.DB_NAME || 'Not configured'}`);
      console.log(`   • Upload Dir: ${uploadDir}`);
      console.log(`   • Session Secret: ${process.env.SESSION_SECRET ? 'Configured' : 'Using fallback'}`);
      console.log('🎉 ===============================================');
    });
  } catch (error) {
    console.error('❌ Failed to start server:', error);
    process.exit(1);
  }
}

// Start the application
startServer();