import express from 'express';
import multer from 'multer';
import path from 'path';
import bcrypt from 'bcryptjs';
import { fileURLToPath } from 'url';
import { storage } from './storage.js';
import { whatsappService } from './services/whatsapp.js';
import { botService } from './services/bot.js';
import { pollingService } from './services/polling.js';
import { openaiService } from './services/openai.js';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

// Configure multer for file uploads
const uploadStorage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, process.env.UPLOAD_DIR || 'uploads/');
  },
  filename: (req, file, cb) => {
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
    cb(null, file.fieldname + '-' + uniqueSuffix + path.extname(file.originalname));
  }
});

const upload = multer({ 
  storage: uploadStorage,
  limits: {
    fileSize: parseInt(process.env.MAX_FILE_SIZE) || 10 * 1024 * 1024 // 10MB default
  }
});

export function setupRoutes(app) {
  const router = express.Router();

  // ===== SYSTEM ENDPOINTS =====
  
  // System status endpoint
  router.get('/system/status', async (req, res) => {
    try {
      const whatsappStatus = await whatsappService.getStateInstance();
      const pollingStatus = pollingService.getStatus();
      const dbStats = await storage.getDashboardStats();
      
      res.json({
        status: 'online',
        whatsapp: whatsappStatus ? 'online' : 'offline',
        polling: pollingStatus.isRunning ? 'running' : 'stopped',
        openai: process.env.OPENAI_API_KEY ? 'configured' : 'not_configured',
        database: 'connected',
        stats: dbStats,
        timestamp: new Date().toISOString(),
        version: '1.0.0'
      });
    } catch (error) {
      console.error('❌ System status error:', error);
      res.status(500).json({
        status: 'error',
        error: error.message
      });
    }
  });

  // Health check for load balancers
  router.get('/health', async (req, res) => {
    res.json({ status: 'healthy', timestamp: new Date().toISOString() });
  });

  // ===== WHATSAPP ENDPOINTS =====

  // WhatsApp webhook endpoint (primary)
  router.post('/green-api/webhook', async (req, res) => {
    try {
      console.log('📨 Webhook received:', JSON.stringify(req.body, null, 2));
      
      const parsedMessage = whatsappService.parseIncomingMessage(req.body);
      
      if (!parsedMessage) {
        return res.json({ status: 'ignored', reason: 'no_parseable_content' });
      }

      console.log('✅ Processing webhook message:', `"${parsedMessage.message}" from ${parsedMessage.chatId}`);
      
      // Process with bot service
      const result = await botService.processMessage(parsedMessage);

      if (result.success) {
        res.json({ 
          status: 'success',
          message: 'Message processed successfully',
          user_id: result.user?.id,
          conversation_id: result.conversation?.id
        });
      } else {
        res.status(500).json({
          status: 'error',
          error: 'Message processing failed',
          details: result.error
        });
      }
    } catch (error) {
      console.error('❌ Webhook error:', error);
      res.status(500).json({
        status: 'error',
        error: 'Failed to process webhook'
      });
    }
  });

  // Send message endpoint (manual testing)
  router.post('/whatsapp/send', async (req, res) => {
    try {
      const { chatId, message, messageType = 'text' } = req.body;
      
      if (!chatId || !message) {
        return res.status(400).json({
          error: 'chatId and message are required'
        });
      }

      const result = await whatsappService.sendMessage(chatId, message, messageType);
      
      if (result.success) {
        // Log the manual send
        await storage.logEvent('info', 'manual_send', 'Manual message sent', {
          chatId, message, messageType, messageId: result.messageId
        });

        res.json({
          status: 'success',
          message: 'Message sent successfully',
          messageId: result.messageId
        });
      } else {
        res.status(500).json({
          status: 'error',
          error: result.error
        });
      }
    } catch (error) {
      console.error('❌ Send message error:', error);
      res.status(500).json({
        error: 'Failed to send message'
      });
    }
  });

  // ===== ANALYTICS ENDPOINTS =====

  // Dashboard statistics
  router.get('/dashboard/stats', async (req, res) => {
    try {
      const tenantId = req.query.tenant_id || process.env.DEFAULT_TENANT_ID;
      const stats = await storage.getDashboardStats(tenantId);
      
      res.json(stats);
    } catch (error) {
      console.error('❌ Dashboard stats error:', error);
      res.status(500).json({ error: 'Failed to fetch dashboard stats' });
    }
  });

  // Keyword analytics
  router.get('/analytics/keywords', async (req, res) => {
    try {
      const tenantId = req.query.tenant_id || process.env.DEFAULT_TENANT_ID;
      const limit = parseInt(req.query.limit) || 10;
      
      const keywords = await storage.getTopKeywords(tenantId, limit);
      
      res.json(keywords);
    } catch (error) {
      console.error('❌ Keywords analytics error:', error);
      res.status(500).json({ error: 'Failed to fetch keyword analytics' });
    }
  });

  // Recent conversations
  router.get('/conversations/recent', async (req, res) => {
    try {
      const tenantId = req.query.tenant_id || process.env.DEFAULT_TENANT_ID;
      const limit = parseInt(req.query.limit) || 10;
      
      const conversations = await storage.getRecentConversations(tenantId, limit);
      
      res.json(conversations);
    } catch (error) {
      console.error('❌ Recent conversations error:', error);
      res.status(500).json({ error: 'Failed to fetch recent conversations' });
    }
  });

  // ===== AUTHENTICATION ENDPOINTS =====

  // Admin login
  router.post('/auth/login', async (req, res) => {
    try {
      const { username, password } = req.body;
      
      if (!username || !password) {
        return res.status(400).json({
          error: 'Username and password are required'
        });
      }

      // Check database first
      const adminUser = await storage.getAdminUser(username);
      
      if (adminUser && await bcrypt.compare(password, adminUser.password)) {
        req.session.isAdmin = true;
        req.session.adminId = adminUser.id;
        req.session.username = adminUser.username;
        
        res.json({
          status: 'success',
          message: 'Login successful',
          user: {
            id: adminUser.id,
            username: adminUser.username,
            email: adminUser.email,
            is_super_admin: adminUser.is_super_admin
          }
        });
      } else if (username === process.env.ADMIN_USERNAME && 
                 password === process.env.ADMIN_PASSWORD) {
        // Fallback to environment variables
        req.session.isAdmin = true;
        req.session.username = username;
        
        res.json({
          status: 'success',
          message: 'Login successful (env auth)',
          user: { username }
        });
      } else {
        res.status(401).json({
          error: 'Invalid credentials'
        });
      }
    } catch (error) {
      console.error('❌ Login error:', error);
      res.status(500).json({
        error: 'Login failed'
      });
    }
  });

  // Admin logout
  router.post('/auth/logout', (req, res) => {
    req.session.destroy((err) => {
      if (err) {
        console.error('❌ Logout error:', err);
        return res.status(500).json({ error: 'Logout failed' });
      }
      res.json({
        status: 'success',
        message: 'Logout successful'
      });
    });
  });

  // Check authentication status
  router.get('/auth/status', (req, res) => {
    res.json({
      isAuthenticated: !!req.session.isAdmin,
      user: req.session.username ? {
        username: req.session.username,
        id: req.session.adminId
      } : null
    });
  });

  // ===== PROTECTED ROUTES MIDDLEWARE =====
  
  const requireAuth = (req, res, next) => {
    if (!req.session.isAdmin) {
      return res.status(401).json({
        error: 'Authentication required'
      });
    }
    next();
  };

  // ===== ADMIN ENDPOINTS =====

  // Admin dashboard data
  router.get('/admin/dashboard', requireAuth, async (req, res) => {
    try {
      const stats = await storage.getDashboardStats();
      const keywords = await storage.getTopKeywords(null, 20);
      const conversations = await storage.getRecentConversations(null, 20);
      
      res.json({
        stats,
        keywords,
        conversations,
        generatedAt: new Date().toISOString()
      });
    } catch (error) {
      console.error('❌ Admin dashboard error:', error);
      res.status(500).json({ error: 'Failed to fetch admin dashboard data' });
    }
  });

  // Business templates
  router.get('/admin/templates', requireAuth, async (req, res) => {
    try {
      const templates = await storage.getBusinessTemplates();
      res.json(templates);
    } catch (error) {
      console.error('❌ Templates error:', error);
      res.status(500).json({ error: 'Failed to fetch templates' });
    }
  });

  // System logs
  router.get('/admin/logs', requireAuth, async (req, res) => {
    try {
      const page = parseInt(req.query.page) || 1;
      const limit = parseInt(req.query.limit) || 50;
      const level = req.query.level;
      const category = req.query.category;
      
      // This would need a proper logs fetching method in storage
      res.json({
        logs: [],
        pagination: { page, limit, total: 0 },
        message: 'Logs endpoint ready for implementation'
      });
    } catch (error) {
      console.error('❌ Logs error:', error);
      res.status(500).json({ error: 'Failed to fetch logs' });
    }
  });

  // ===== FILE UPLOAD ENDPOINTS =====

  // File upload
  router.post('/upload', requireAuth, upload.single('file'), async (req, res) => {
    try {
      if (!req.file) {
        return res.status(400).json({
          error: 'No file uploaded'
        });
      }

      // Save file record to database
      const fileId = await storage.createFile({
        filename: req.file.filename,
        original_name: req.file.originalname,
        file_path: req.file.path,
        file_size: req.file.size,
        mime_type: req.file.mimetype,
        file_type: this.getFileType(req.file.mimetype),
        uploaded_by: req.session.adminId
      });

      res.json({
        status: 'success',
        file: {
          id: fileId,
          filename: req.file.filename,
          originalname: req.file.originalname,
          size: req.file.size,
          path: req.file.path
        }
      });
    } catch (error) {
      console.error('❌ Upload error:', error);
      res.status(500).json({
        error: 'Failed to upload file'
      });
    }
  });

  // ===== POLLING CONTROL ENDPOINTS =====

  // Polling status
  router.get('/polling/status', requireAuth, (req, res) => {
    const status = pollingService.getStatus();
    res.json(status);
  });

  // Start/stop polling
  router.post('/polling/control', requireAuth, async (req, res) => {
    try {
      const { action } = req.body;
      
      if (action === 'start') {
        await pollingService.start();
        res.json({ status: 'success', message: 'Polling started' });
      } else if (action === 'stop') {
        await pollingService.stop();
        res.json({ status: 'success', message: 'Polling stopped' });
      } else if (action === 'poll_once') {
        await pollingService.pollOnce();
        res.json({ status: 'success', message: 'Manual poll triggered' });
      } else {
        res.status(400).json({ error: 'Invalid action. Use: start, stop, or poll_once' });
      }
    } catch (error) {
      console.error('❌ Polling control error:', error);
      res.status(500).json({ error: 'Failed to control polling' });
    }
  });

  // ===== TESTING ENDPOINTS =====

  // Test OpenAI
  router.post('/test/openai', requireAuth, async (req, res) => {
    try {
      const { message } = req.body;
      
      if (!message) {
        return res.status(400).json({ error: 'Message is required' });
      }

      const result = await openaiService.processMessage(message);
      res.json(result);
    } catch (error) {
      console.error('❌ OpenAI test error:', error);
      res.status(500).json({ error: 'OpenAI test failed' });
    }
  });

  // Test WhatsApp
  router.post('/test/whatsapp', requireAuth, async (req, res) => {
    try {
      const status = await whatsappService.getStateInstance();
      res.json({ 
        status: status ? 'authorized' : 'unauthorized',
        instance_id: process.env.GREEN_API_INSTANCE_ID
      });
    } catch (error) {
      console.error('❌ WhatsApp test error:', error);
      res.status(500).json({ error: 'WhatsApp test failed' });
    }
  });

  // Helper method for file type detection
  function getFileType(mimeType) {
    if (mimeType.startsWith('image/')) return 'image';
    if (mimeType.startsWith('video/')) return 'video';
    if (mimeType.startsWith('audio/')) return 'audio';
    if (mimeType.includes('pdf') || mimeType.includes('document') || 
        mimeType.includes('text') || mimeType.includes('excel') || 
        mimeType.includes('word')) return 'document';
    return 'other';
  }

  // Apply router to app
  app.use('/api', router);

  console.log('✅ API routes configured successfully');
}

export default setupRoutes;