import { storage } from '../storage.js';
import { openaiService } from './openai.js';
import { whatsappService } from './whatsapp.js';

export class BotService {
  constructor() {
    this.storage = storage;
    this.ai = openaiService;
    this.whatsapp = whatsappService;
    console.log('✅ Bot service initialized');
  }

  async processMessage(parsedMessage) {
    try {
      console.log('🤖 Processing message:', parsedMessage.message);
      
      // Get or create user
      let user = await this.storage.getUser(parsedMessage.chatId);
      if (!user) {
        user = await this.storage.createUser({
          phone_number: this.extractPhoneNumber(parsedMessage.chatId),
          chat_id: parsedMessage.chatId,
          name: parsedMessage.senderName,
          profile_name: parsedMessage.profileName
        });
        console.log('👤 Created new user:', user.id);
      } else {
        // Update last activity
        await this.storage.updateUser(parsedMessage.chatId, {
          last_activity: new Date(),
          name: parsedMessage.senderName || user.name,
          profile_name: parsedMessage.profileName || user.profile_name
        });
      }

      // Get or create conversation
      let conversation = await this.storage.getConversation(user.id);
      if (!conversation) {
        conversation = await this.storage.createConversation({
          user_id: user.id,
          title: `Conversation with ${user.name || 'User'}`,
          language: user.language || 'ms'
        });
        console.log('💬 Created new conversation:', conversation.id);
      } else {
        // Check if conversation needs reactivation (24+ hours inactive)
        const lastMessageTime = new Date(conversation.last_message_at);
        const hoursSinceLastMessage = (Date.now() - lastMessageTime.getTime()) / (1000 * 60 * 60);
        
        if (hoursSinceLastMessage > 24) {
          console.log(`⏰ User inactive for ${hoursSinceLastMessage.toFixed(1)} hours, reactivating conversation`);
          
          // Update conversation status
          await this.storage.updateConversation(conversation.id, {
            status: 'active',
            last_message_at: new Date()
          });
          
          // Send reactivation message if it's a help request
          if (this.whatsapp.isHelpRequest(parsedMessage.message)) {
            return await this.handleHelpRequest(parsedMessage, user, conversation);
          }
        }
      }

      // Save user message
      await this.storage.createMessage({
        conversation_id: conversation.id,
        content: parsedMessage.message,
        sender_type: 'user',
        message_type: parsedMessage.messageType || 'text'
      });

      // Get available keywords for this tenant
      const keywords = await this.storage.getKeywords(user.tenant_id, user.language);
      
      // Check for help requests
      if (this.whatsapp.isHelpRequest(parsedMessage.message)) {
        return await this.handleHelpRequest(parsedMessage, user, conversation, keywords);
      }

      // Extract and track keywords
      const foundKeywords = this.whatsapp.extractKeywords(parsedMessage.message, keywords);
      for (const keyword of foundKeywords) {
        await this.storage.trackKeywordMention(keyword, user.id, user.tenant_id);
        console.log(`🔍 Tracked keyword: ${keyword}`);
      }

      // Process with AI
      const recentMessages = await this.storage.getRecentMessages(conversation.id, 10);
      const conversationHistory = recentMessages.map(msg => ({
        role: msg.sender_type === 'user' ? 'user' : 'assistant',
        content: msg.content
      }));

      const aiResponse = await this.ai.processMessage(
        parsedMessage.message,
        { user, conversation, keywords },
        conversationHistory
      );

      // Save bot response
      await this.storage.createMessage({
        conversation_id: conversation.id,
        content: aiResponse.response,
        sender_type: 'bot',
        message_type: 'text',
        metadata: {
          intent: aiResponse.intent,
          keywords: aiResponse.keywords,
          category: aiResponse.category
        }
      });

      // Send response
      const sendResult = await this.whatsapp.sendMessage(
        parsedMessage.chatId,
        aiResponse.response
      );

      if (sendResult.success) {
        console.log('✅ Bot response sent successfully');
      } else {
        console.error('❌ Failed to send bot response:', sendResult.error);
      }

      // Log the interaction
      await this.storage.logEvent(
        'info',
        'bot_interaction',
        'Message processed successfully',
        {
          userId: user.id,
          conversationId: conversation.id,
          messageLength: parsedMessage.message.length,
          responseLength: aiResponse.response.length,
          intent: aiResponse.intent,
          keywordsFound: foundKeywords
        },
        user.tenant_id,
        user.id
      );

      return {
        success: true,
        response: aiResponse.response,
        user: user,
        conversation: conversation
      };

    } catch (error) {
      console.error('❌ Error processing message:', error);
      
      // Send error message to user
      await this.whatsapp.sendMessage(
        parsedMessage.chatId,
        'Maaf, sistem sedang menghadapi masalah teknikal. Sila cuba sebentar lagi. 🔧'
      );

      return {
        success: false,
        error: error.message
      };
    }
  }

  async handleHelpRequest(parsedMessage, user, conversation, keywords = []) {
    try {
      console.log('🎯 [HELP] Help request detected, generating keyword list');
      
      // Get fresh keywords if not provided
      if (!keywords || keywords.length === 0) {
        keywords = await this.storage.getKeywords(user.tenant_id, user.language);
      }

      const keywordList = this.whatsapp.generateKeywordList(keywords, user.language);
      
      console.log(`🎯 [HELP] Generated keyword list with ${keywords.length} keywords`);

      // Save help request message
      await this.storage.createMessage({
        conversation_id: conversation.id,
        content: parsedMessage.message,
        sender_type: 'user',
        message_type: 'text'
      });

      // Save help response
      await this.storage.createMessage({
        conversation_id: conversation.id,
        content: keywordList,
        sender_type: 'bot',
        message_type: 'text',
        metadata: {
          intent: 'help_request',
          category: 'support',
          keywords_shown: keywords.length
        }
      });

      // Send keyword list
      const sendResult = await this.whatsapp.sendMessage(
        parsedMessage.chatId,
        keywordList
      );

      if (sendResult.success) {
        console.log('✅ Help response sent successfully');
      } else {
        console.error('❌ Failed to send help response:', sendResult.error);
      }

      return {
        success: true,
        response: keywordList,
        type: 'help',
        keywordsShown: keywords.length
      };

    } catch (error) {
      console.error('❌ Error handling help request:', error);
      
      const fallbackMessage = user.language === 'ms' 
        ? 'Maaf, sistem bantuan tidak tersedia pada masa ini. Sila cuba lagi kemudian.'
        : 'Sorry, help system is not available right now. Please try again later.';
        
      await this.whatsapp.sendMessage(parsedMessage.chatId, fallbackMessage);
      
      return {
        success: false,
        error: error.message
      };
    }
  }

  extractPhoneNumber(chatId) {
    // Extract phone number from chatId (remove @c.us)
    return chatId.replace('@c.us', '').replace(/\D/g, '');
  }

  async analyzeUserIntent(message, userId) {
    try {
      const analysis = await this.ai.analyzeIntent(message);
      
      // Store analysis results if needed
      if (analysis.category !== 'unclear') {
        await this.storage.logEvent(
          'info',
          'intent_analysis',
          'User intent analyzed',
          {
            userId: userId,
            analysis: analysis,
            message_preview: message.substring(0, 100)
          }
        );
      }

      return analysis;
    } catch (error) {
      console.error('❌ Error analyzing user intent:', error);
      return null;
    }
  }

  async updateUserDemographics(userId, messageHistory) {
    try {
      const demographics = await this.ai.extractDemographics(messageHistory);
      
      if (demographics.confidence_score > 0.3) {
        // Update user demographics in database
        // This would require a demographics table update method
        console.log('📊 Extracted demographics:', demographics);
      }

      return demographics;
    } catch (error) {
      console.error('❌ Error updating demographics:', error);
      return null;
    }
  }

  async generateBusinessReport(tenantId, dateRange = null) {
    try {
      const stats = await this.storage.getDashboardStats(tenantId);
      const topKeywords = await this.storage.getTopKeywords(tenantId, 20);
      const recentConversations = await this.storage.getRecentConversations(tenantId, 50);

      return {
        stats: stats,
        keywords: topKeywords,
        conversations: recentConversations,
        generatedAt: new Date().toISOString()
      };
    } catch (error) {
      console.error('❌ Error generating business report:', error);
      return null;
    }
  }
}

export const botService = new BotService();