import OpenAI from 'openai';

export class OpenAIService {
  constructor() {
    this.client = new OpenAI({
      apiKey: process.env.OPENAI_API_KEY
    });
    
    if (!process.env.OPENAI_API_KEY) {
      console.warn('⚠️ OpenAI API key not configured');
    } else {
      console.log('✅ OpenAI service initialized');
    }
  }

  async processMessage(message, userContext = {}, conversationHistory = []) {
    try {
      const systemPrompt = this.buildSystemPrompt(userContext);
      
      const messages = [
        { role: 'system', content: systemPrompt },
        ...conversationHistory.slice(-10), // Last 10 messages for context
        { role: 'user', content: message }
      ];

      const completion = await this.client.chat.completions.create({
        model: 'gpt-4o', // Latest model as of knowledge cutoff
        messages: messages,
        max_tokens: 1000,
        temperature: 0.7,
        response_format: { type: 'json_object' }
      });

      const response = JSON.parse(completion.choices[0].message.content);
      
      return {
        response: response.response || 'Maaf, saya tidak faham. Boleh cuba lagi?',
        intent: response.intent || 'unknown',
        keywords: response.keywords || [],
        needsFollowUp: response.needsFollowUp || false,
        category: response.category || 'general'
      };
    } catch (error) {
      console.error('❌ OpenAI processing error:', error);
      return {
        response: 'Maaf, sistem sedang menghadapi masalah teknikal. Sila cuba sebentar lagi.',
        intent: 'error',
        keywords: [],
        needsFollowUp: false,
        category: 'system'
      };
    }
  }

  async analyzeIntent(message, surveyContext = {}) {
    try {
      const prompt = `
Analyze this message for business survey intent. Return JSON with:
- category: "high_potential", "medium_potential", "low_potential", "not_eligible", "unclear"
- business_interest_score: 0-100
- urgency_level: "immediate", "soon", "future", "none"
- eligibility_status: "eligible", "maybe_eligible", "not_eligible", "unknown"
- keywords: array of relevant keywords found
- reasoning: brief explanation

Message: "${message}"
Survey Context: ${JSON.stringify(surveyContext)}

Respond in JSON format only.
      `;

      const completion = await this.client.chat.completions.create({
        model: 'gpt-4o',
        messages: [{ role: 'user', content: prompt }],
        max_tokens: 500,
        temperature: 0.3,
        response_format: { type: 'json_object' }
      });

      return JSON.parse(completion.choices[0].message.content);
    } catch (error) {
      console.error('❌ Intent analysis error:', error);
      return {
        category: 'unclear',
        business_interest_score: 0,
        urgency_level: 'none',
        eligibility_status: 'unknown',
        keywords: [],
        reasoning: 'Analysis failed due to technical error'
      };
    }
  }

  async extractDemographics(messageHistory = []) {
    try {
      const prompt = `
Extract demographic information from this conversation history. Return JSON with:
- age_range: "18-25", "26-35", "36-45", "46-55", "56-65", "65+" or null
- gender: "male", "female", "other", "unknown"
- race: Malaysian races like "Malay", "Chinese", "Indian", "Other" or null
- occupation: job/profession or null
- location: city/state in Malaysia or null
- income_range: "Below RM2000", "RM2000-5000", "RM5000-10000", "Above RM10000" or null
- education_level: "SPM", "Diploma", "Degree", "Masters", "PhD" or null
- family_size: number or null
- confidence_score: 0.0-1.0 (how confident you are in the extraction)

Messages: ${JSON.stringify(messageHistory.slice(-20))}

Respond in JSON format only. Use null for unknown values.
      `;

      const completion = await this.client.chat.completions.create({
        model: 'gpt-4o',
        messages: [{ role: 'user', content: prompt }],
        max_tokens: 300,
        temperature: 0.1,
        response_format: { type: 'json_object' }
      });

      return JSON.parse(completion.choices[0].message.content);
    } catch (error) {
      console.error('❌ Demographics extraction error:', error);
      return {
        age_range: null,
        gender: 'unknown',
        race: null,
        occupation: null,
        location: null,
        income_range: null,
        education_level: null,
        family_size: null,
        confidence_score: 0.0
      };
    }
  }

  async generateKeywordResponse(keyword, businessContext = {}) {
    try {
      const prompt = `
Generate a helpful response for the keyword "${keyword}" in the context of this business:
${JSON.stringify(businessContext)}

The response should be:
- Friendly and professional in Malaysian style
- Include relevant emoji
- Provide useful information
- End with a call-to-action if appropriate
- Maximum 200 words
- In Bahasa Malaysia if the business serves local customers

Return JSON with:
- response: the generated response text
- category: "product", "service", "support", "pricing", "general"
- suggested_followup: optional follow-up question

Respond in JSON format only.
      `;

      const completion = await this.client.chat.completions.create({
        model: 'gpt-4o',
        messages: [{ role: 'user', content: prompt }],
        max_tokens: 400,
        temperature: 0.8,
        response_format: { type: 'json_object' }
      });

      return JSON.parse(completion.choices[0].message.content);
    } catch (error) {
      console.error('❌ Keyword response generation error:', error);
      return {
        response: `Terima kasih atas pertanyaan mengenai "${keyword}". Sila hubungi kami untuk maklumat lanjut.`,
        category: 'general',
        suggested_followup: null
      };
    }
  }

  buildSystemPrompt(userContext = {}) {
    return `
Anda adalah AI assistant untuk sistem chatbot WhatsApp survey perniagaan di Malaysia. 

ARAHAN UTAMA:
- Selalu respond dalam Bahasa Malaysia kecuali diminta sebaliknya
- Gunakan gaya komunikasi yang mesra dan professional
- Guna emoji yang sesuai untuk menarik perhatian
- Fokus pada membantu pengguna dengan survey atau maklumat perniagaan
- Jika ada soalan luar topik, redirect dengan sopan ke tujuan utama

USER CONTEXT: ${JSON.stringify(userContext)}

RESPONSE FORMAT: Respond in JSON with these keys:
- response: your main response text
- intent: detected user intent
- keywords: array of keywords found in user message
- needsFollowUp: boolean if follow-up needed
- category: message category (product, service, support, pricing, general)

Pastikan JSON format adalah valid dan lengkap.
    `;
  }

  async generateWelcomeMessage(businessInfo = {}) {
    try {
      const prompt = `
Generate a welcoming WhatsApp message for this business: ${JSON.stringify(businessInfo)}

The message should:
- Be warm and professional
- Include business name and key info
- Mention available services briefly
- Encourage engagement
- Use appropriate emoji
- Be in Bahasa Malaysia
- Maximum 150 words

Return JSON with:
- message: the welcome message text
- suggested_actions: array of suggested user actions

Respond in JSON format only.
      `;

      const completion = await this.client.chat.completions.create({
        model: 'gpt-4o',
        messages: [{ role: 'user', content: prompt }],
        max_tokens: 300,
        temperature: 0.7,
        response_format: { type: 'json_object' }
      });

      return JSON.parse(completion.choices[0].message.content);
    } catch (error) {
      console.error('❌ Welcome message generation error:', error);
      return {
        message: '🙋‍♀️ Selamat datang! Saya di sini untuk membantu anda. Boleh beritahu saya apa yang anda perlukan?',
        suggested_actions: ['Tanya soalan', 'Lihat perkhidmatan', 'Hubungi kami']
      };
    }
  }
}

export const openaiService = new OpenAIService();