import { whatsappService } from './whatsapp.js';
import { botService } from './bot.js';

export class PollingService {
  constructor() {
    this.whatsappService = whatsappService;
    this.botService = botService;
    this.isRunning = false;
    this.pollingInterval = parseInt(process.env.POLLING_INTERVAL) || 10000; // 10 seconds
    this.intervalId = null;
    this.processedMessages = new Set(); // Track processed messages
  }

  async start() {
    if (this.isRunning) {
      console.log('⚠️ Polling service already running');
      return;
    }

    this.isRunning = true;
    console.log('🔄 Starting polling service...');
    
    this.intervalId = setInterval(async () => {
      await this.pollMessages();
    }, this.pollingInterval);

    console.log(`✅ Polling service started with ${this.pollingInterval/1000}s interval`);
  }

  async stop() {
    if (!this.isRunning) {
      return;
    }

    this.isRunning = false;
    
    if (this.intervalId) {
      clearInterval(this.intervalId);
      this.intervalId = null;
    }

    console.log('⏹️ Polling service stopped');
  }

  async pollMessages() {
    try {
      const notification = await this.whatsappService.receiveNotification();
      
      if (!notification) {
        return; // No messages
      }

      const receiptId = notification.receiptId;
      const webhookBody = notification.body;

      console.log('📨 New notification received via polling:', receiptId);

      // Check for duplicate processing
      if (this.processedMessages.has(receiptId)) {
        console.log('🔄 Duplicate message detected, skipping:', receiptId);
        await this.whatsappService.deleteNotification(receiptId);
        return;
      }

      // Mark as processed
      this.processedMessages.add(receiptId);

      // Clean up old processed message IDs (keep last 1000)
      if (this.processedMessages.size > 1000) {
        const oldIds = Array.from(this.processedMessages).slice(0, 500);
        oldIds.forEach(id => this.processedMessages.delete(id));
      }

      // Skip outgoing messages
      if (webhookBody.typeWebhook === 'outgoingMessageStatus' || 
          webhookBody.typeWebhook === 'outgoingAPIMessageStatus') {
        console.log('🔇 Skipping outgoing message:', webhookBody.typeWebhook);
        await this.whatsappService.deleteNotification(receiptId);
        return;
      }

      // Parse the message
      const parsedMessage = this.whatsappService.parseIncomingMessage(webhookBody);
      
      if (!parsedMessage) {
        console.log('🔇 No parseable message content');
        await this.whatsappService.deleteNotification(receiptId);
        return;
      }

      console.log('✅ UNIVERSAL POLLING: Processing user message from sender:', 
        parsedMessage.sender, 
        '(bot WIDs:', this.whatsappService.instanceId + '@c.us', ')'
      );

      // Process the message with bot service
      console.log('📱 Processing polled message:', `"${parsedMessage.message}" from ${parsedMessage.chatId}`);
      
      const result = await this.botService.processMessage(parsedMessage);
      
      if (result.success) {
        console.log('📝 POLLING RESPONSE TRACKED:', parsedMessage.chatId, '- successful');
      } else {
        console.error('❌ POLLING RESPONSE FAILED:', result.error);
      }

      // Delete the notification after processing
      await this.whatsappService.deleteNotification(receiptId);
      
    } catch (error) {
      console.error('❌ Polling error:', error);
    }
  }

  getStatus() {
    return {
      isRunning: this.isRunning,
      interval: this.pollingInterval,
      processedCount: this.processedMessages.size
    };
  }

  // Method to manually trigger polling (for testing)
  async pollOnce() {
    console.log('🔄 Manual poll triggered');
    await this.pollMessages();
  }
}

export const pollingService = new PollingService();