export class WhatsAppService {
  constructor() {
    this.instanceId = process.env.GREEN_API_INSTANCE_ID;
    this.accessToken = process.env.GREEN_API_ACCESS_TOKEN;
    this.baseUrl = `https://api.green-api.com/waInstance${this.instanceId}`;
    this.botPhoneNumbers = new Set([
      `${this.instanceId}@c.us`,
      `60${this.instanceId}@c.us` // Common Malaysian format
    ]);
    
    if (!this.instanceId || !this.accessToken) {
      console.warn('⚠️ WhatsApp credentials not configured');
    } else {
      console.log('✅ WhatsApp service configured for instance:', this.instanceId);
    }
  }

  async sendMessage(chatId, message, messageType = 'text') {
    try {
      const formattedChatId = this.formatPhoneNumber(chatId);
      let endpoint = 'sendMessage';
      let payload = {
        chatId: formattedChatId,
        message: message
      };

      // Handle different message types
      if (messageType === 'interactive') {
        endpoint = 'sendMessage';
        payload.message = message; // message should contain interactive elements
      }

      const response = await fetch(`${this.baseUrl}/${endpoint}/${this.accessToken}`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(payload),
      });

      if (!response.ok) {
        const errorData = await response.text();
        throw new Error(`WhatsApp API error ${response.status}: ${errorData}`);
      }

      const result = await response.json();
      console.log('✅ Message sent successfully:', result.idMessage);
      return {
        success: true,
        messageId: result.idMessage,
        data: result
      };
    } catch (error) {
      console.error('❌ Failed to send WhatsApp message:', error);
      return {
        success: false,
        error: error.message
      };
    }
  }

  async sendButtonMessage(chatId, text, buttons) {
    try {
      const buttonMessage = {
        text: text,
        footer: "Pilih salah satu pilihan:",
        buttons: buttons.map((btn, index) => ({
          type: "reply",
          reply: {
            id: `btn_${index}`,
            title: btn.title || btn
          }
        }))
      };

      return await this.sendMessage(chatId, buttonMessage, 'interactive');
    } catch (error) {
      console.error('❌ Failed to send button message:', error);
      return { success: false, error: error.message };
    }
  }

  async sendListMessage(chatId, text, sections) {
    try {
      const listMessage = {
        text: text,
        buttonText: "Lihat Pilihan",
        sections: sections.map(section => ({
          title: section.title,
          rows: section.rows.map((row, index) => ({
            id: `list_${index}`,
            title: row.title || row,
            description: row.description || ""
          }))
        }))
      };

      return await this.sendMessage(chatId, listMessage, 'interactive');
    } catch (error) {
      console.error('❌ Failed to send list message:', error);
      return { success: false, error: error.message };
    }
  }

  async receiveNotification() {
    try {
      const response = await fetch(`${this.baseUrl}/receiveNotification/${this.accessToken}`, {
        method: 'GET',
      });

      if (!response.ok) {
        if (response.status === 404) {
          return null; // No messages
        }
        throw new Error(`WhatsApp API error: ${response.status}`);
      }

      const result = await response.json();
      return result;
    } catch (error) {
      console.error('❌ Failed to receive notifications:', error);
      return null;
    }
  }

  async deleteNotification(receiptId) {
    try {
      const response = await fetch(`${this.baseUrl}/deleteNotification/${this.accessToken}/${receiptId}`, {
        method: 'DELETE',
      });

      return response.ok;
    } catch (error) {
      console.error('❌ Failed to delete notification:', error);
      return false;
    }
  }

  async getStateInstance() {
    try {
      const response = await fetch(`${this.baseUrl}/getStateInstance/${this.accessToken}`);
      
      if (!response.ok) {
        throw new Error(`WhatsApp API error: ${response.status}`);
      }

      const result = await response.json();
      return result.stateInstance === 'authorized';
    } catch (error) {
      console.error('❌ Failed to check WhatsApp state:', error);
      return false;
    }
  }

  parseIncomingMessage(webhookData) {
    try {
      // Skip non-message webhooks
      if (webhookData.typeWebhook !== 'incomingMessageReceived') {
        return null;
      }

      const senderData = webhookData.senderData;
      const messageData = webhookData.messageData;
      
      if (!senderData || !messageData) {
        console.log('🔇 Missing sender or message data');
        return null;
      }

      // Check if message is from bot itself
      const senderWid = webhookData.instanceData?.wid;
      const senderChatId = senderData.sender || senderData.chatId;
      
      if (this.isBotMessage(senderChatId, senderWid)) {
        console.log('🔇 Skipping bot message:', senderChatId);
        return null;
      }

      // Extract message text
      let messageText = '';
      
      if (messageData.textMessageData?.textMessage) {
        messageText = messageData.textMessageData.textMessage;
      } else if (messageData.extendedTextMessageData?.text) {
        messageText = messageData.extendedTextMessageData.text;
      } else if (messageData.buttonResponseData?.selectedButtonId) {
        messageText = messageData.buttonResponseData.selectedButtonId;
      } else if (messageData.listResponseData?.singleSelectReply?.selectedRowId) {
        messageText = messageData.listResponseData.singleSelectReply.selectedRowId;
      }

      if (!messageText || messageText.trim() === '') {
        console.log('🔇 No extractable text from message');
        return null;
      }

      return {
        chatId: senderData.chatId,
        sender: senderData.sender,
        message: messageText.trim(),
        senderName: senderData.senderName || senderData.chatName || 'Unknown',
        profileName: senderData.senderContactName || senderData.senderName,
        messageType: messageData.typeMessage,
        timestamp: webhookData.timestamp || Date.now(),
        messageId: webhookData.idMessage
      };
    } catch (error) {
      console.error('❌ Error parsing message:', error);
      return null;
    }
  }

  isBotMessage(senderChatId, senderWid) {
    // Check various bot identifier formats
    const checkIds = [
      senderChatId,
      senderWid,
      this.instanceId + '@c.us',
      '60' + this.instanceId + '@c.us'
    ].filter(Boolean);

    return checkIds.some(id => this.botPhoneNumbers.has(id));
  }

  formatPhoneNumber(phoneNumber) {
    // Remove any non-digit characters except @
    let cleaned = phoneNumber.replace(/[^\d@]/g, '');
    
    // If it doesn't end with @c.us, add it
    if (!cleaned.includes('@')) {
      cleaned += '@c.us';
    }
    
    return cleaned;
  }

  // Generate keyword list for help responses
  generateKeywordList(keywords, language = 'ms') {
    if (!keywords || keywords.length === 0) {
      return language === 'ms' 
        ? 'Tiada kata kunci tersedia pada masa ini.'
        : 'No keywords available at this time.';
    }

    const intro = language === 'ms' 
      ? '📋 *Kata Kunci Tersedia:*\n\n'
      : '📋 *Available Keywords:*\n\n';

    const keywordList = keywords
      .slice(0, 10) // Limit to top 10
      .map(k => `• ${k.keyword}`)
      .join('\n');

    const footer = language === 'ms'
      ? '\n\n💬 Taip mana-mana kata kunci untuk maklumat lanjut!'
      : '\n\n💬 Type any keyword for more information!';

    return intro + keywordList + footer;
  }

  // Check if message is a help request
  isHelpRequest(message) {
    const helpKeywords = [
      'help', 'bantuan', 'tolong', 'panduan', 'guide',
      'list', 'senarai', 'menu', 'pilihan', 'options'
    ];
    
    const lowerMessage = message.toLowerCase();
    return helpKeywords.some(keyword => lowerMessage.includes(keyword));
  }

  // Extract keywords from message for tracking
  extractKeywords(message, availableKeywords = []) {
    const foundKeywords = [];
    const lowerMessage = message.toLowerCase();
    
    availableKeywords.forEach(keyword => {
      if (lowerMessage.includes(keyword.keyword.toLowerCase())) {
        foundKeywords.push(keyword.keyword);
      }
    });
    
    return foundKeywords;
  }
}

export const whatsappService = new WhatsAppService();