import { pool } from './db.js';
import bcrypt from 'bcryptjs';
import { v4 as uuidv4 } from 'uuid';

export class DatabaseStorage {
  constructor() {
    this.db = pool;
  }

  // User management
  async getUser(chatId, tenantId = null) {
    try {
      const defaultTenantId = tenantId || process.env.DEFAULT_TENANT_ID;
      const [rows] = await this.db.execute(
        'SELECT * FROM users WHERE chat_id = ? AND tenant_id = ?',
        [chatId, defaultTenantId]
      );
      return rows[0] || null;
    } catch (error) {
      console.error('Error getting user:', error);
      return null;
    }
  }

  async createUser(userData) {
    try {
      const id = uuidv4();
      const defaultTenantId = userData.tenant_id || process.env.DEFAULT_TENANT_ID;
      
      await this.db.execute(
        `INSERT INTO users (id, tenant_id, phone_number, chat_id, name, profile_name, language) 
         VALUES (?, ?, ?, ?, ?, ?, ?)`,
        [
          id,
          defaultTenantId,
          userData.phone_number,
          userData.chat_id,
          userData.name || null,
          userData.profile_name || null,
          userData.language || 'ms'
        ]
      );

      return await this.getUser(userData.chat_id, defaultTenantId);
    } catch (error) {
      console.error('Error creating user:', error);
      throw error;
    }
  }

  async updateUser(chatId, updates) {
    try {
      const setParts = [];
      const values = [];
      
      Object.entries(updates).forEach(([key, value]) => {
        setParts.push(`${key} = ?`);
        values.push(value);
      });
      
      values.push(chatId);
      
      await this.db.execute(
        `UPDATE users SET ${setParts.join(', ')}, updated_at = CURRENT_TIMESTAMP WHERE chat_id = ?`,
        values
      );
      
      return await this.getUser(chatId);
    } catch (error) {
      console.error('Error updating user:', error);
      throw error;
    }
  }

  // Conversation management
  async getConversation(userId, tenantId = null) {
    try {
      const defaultTenantId = tenantId || process.env.DEFAULT_TENANT_ID;
      const [rows] = await this.db.execute(
        'SELECT * FROM conversations WHERE user_id = ? AND tenant_id = ? ORDER BY created_at DESC LIMIT 1',
        [userId, defaultTenantId]
      );
      return rows[0] || null;
    } catch (error) {
      console.error('Error getting conversation:', error);
      return null;
    }
  }

  async createConversation(conversationData) {
    try {
      const id = uuidv4();
      const defaultTenantId = conversationData.tenant_id || process.env.DEFAULT_TENANT_ID;
      
      await this.db.execute(
        `INSERT INTO conversations (id, user_id, tenant_id, survey_id, title, status, language) 
         VALUES (?, ?, ?, ?, ?, ?, ?)`,
        [
          id,
          conversationData.user_id,
          defaultTenantId,
          conversationData.survey_id || null,
          conversationData.title || 'New Conversation',
          conversationData.status || 'active',
          conversationData.language || 'ms'
        ]
      );

      return await this.getConversationById(id);
    } catch (error) {
      console.error('Error creating conversation:', error);
      throw error;
    }
  }

  async getConversationById(id) {
    try {
      const [rows] = await this.db.execute(
        'SELECT * FROM conversations WHERE id = ?',
        [id]
      );
      return rows[0] || null;
    } catch (error) {
      console.error('Error getting conversation by ID:', error);
      return null;
    }
  }

  async updateConversation(id, updates) {
    try {
      const setParts = [];
      const values = [];
      
      Object.entries(updates).forEach(([key, value]) => {
        setParts.push(`${key} = ?`);
        values.push(value);
      });
      
      values.push(id);
      
      await this.db.execute(
        `UPDATE conversations SET ${setParts.join(', ')}, updated_at = CURRENT_TIMESTAMP WHERE id = ?`,
        values
      );
      
      return await this.getConversationById(id);
    } catch (error) {
      console.error('Error updating conversation:', error);
      throw error;
    }
  }

  // Message management
  async createMessage(messageData) {
    try {
      const id = uuidv4();
      
      await this.db.execute(
        `INSERT INTO messages (id, conversation_id, content, sender_type, message_type, metadata) 
         VALUES (?, ?, ?, ?, ?, ?)`,
        [
          id,
          messageData.conversation_id,
          messageData.content,
          messageData.sender_type,
          messageData.message_type || 'text',
          messageData.metadata ? JSON.stringify(messageData.metadata) : null
        ]
      );

      // Update conversation last_message_at
      await this.db.execute(
        'UPDATE conversations SET last_message_at = CURRENT_TIMESTAMP WHERE id = ?',
        [messageData.conversation_id]
      );

      return id;
    } catch (error) {
      console.error('Error creating message:', error);
      throw error;
    }
  }

  async getRecentMessages(conversationId, limit = 50) {
    try {
      const [rows] = await this.db.execute(
        'SELECT * FROM messages WHERE conversation_id = ? ORDER BY created_at DESC LIMIT ?',
        [conversationId, limit]
      );
      return rows.reverse(); // Return in chronological order
    } catch (error) {
      console.error('Error getting recent messages:', error);
      return [];
    }
  }

  // Keywords management
  async getKeywords(tenantId = null, language = 'ms') {
    try {
      const defaultTenantId = tenantId || process.env.DEFAULT_TENANT_ID;
      const [rows] = await this.db.execute(
        'SELECT * FROM keywords WHERE tenant_id = ? AND language = ? AND is_active = TRUE',
        [defaultTenantId, language]
      );
      return rows;
    } catch (error) {
      console.error('Error getting keywords:', error);
      return [];
    }
  }

  async trackKeywordMention(keyword, userId = null, tenantId = null) {
    try {
      const defaultTenantId = tenantId || process.env.DEFAULT_TENANT_ID;
      
      // Check if keyword mention exists
      const [existing] = await this.db.execute(
        'SELECT * FROM keyword_mentions WHERE keyword = ? AND tenant_id = ?',
        [keyword.toLowerCase(), defaultTenantId]
      );

      if (existing.length > 0) {
        // Update existing
        await this.db.execute(
          'UPDATE keyword_mentions SET count = count + 1, last_mentioned = CURRENT_TIMESTAMP WHERE id = ?',
          [existing[0].id]
        );
        return existing[0];
      } else {
        // Create new
        const id = uuidv4();
        await this.db.execute(
          `INSERT INTO keyword_mentions (id, tenant_id, keyword, count, channel, last_mentioned) 
           VALUES (?, ?, ?, 1, 'whatsapp', CURRENT_TIMESTAMP)`,
          [id, defaultTenantId, keyword.toLowerCase()]
        );
        
        const [newRecord] = await this.db.execute(
          'SELECT * FROM keyword_mentions WHERE id = ?',
          [id]
        );
        return newRecord[0];
      }
    } catch (error) {
      console.error('Error tracking keyword mention:', error);
      throw error;
    }
  }

  async getTopKeywords(tenantId = null, limit = 10) {
    try {
      const defaultTenantId = tenantId || process.env.DEFAULT_TENANT_ID;
      const [rows] = await this.db.execute(
        'SELECT * FROM keyword_mentions WHERE tenant_id = ? ORDER BY count DESC LIMIT ?',
        [defaultTenantId, limit]
      );
      return rows;
    } catch (error) {
      console.error('Error getting top keywords:', error);
      return [];
    }
  }

  // Analytics
  async getDashboardStats(tenantId = null) {
    try {
      const defaultTenantId = tenantId || process.env.DEFAULT_TENANT_ID;
      
      // Active conversations
      const [activeConversations] = await this.db.execute(
        'SELECT COUNT(*) as count FROM conversations WHERE tenant_id = ? AND status = "active"',
        [defaultTenantId]
      );

      // Total users
      const [totalUsers] = await this.db.execute(
        'SELECT COUNT(*) as count FROM users WHERE tenant_id = ?',
        [defaultTenantId]
      );

      // Messages today
      const [messagesToday] = await this.db.execute(
        `SELECT COUNT(*) as count FROM messages m 
         JOIN conversations c ON m.conversation_id = c.id 
         WHERE c.tenant_id = ? AND DATE(m.created_at) = CURDATE()`,
        [defaultTenantId]
      );

      // Keywords tracked
      const [keywordsCount] = await this.db.execute(
        'SELECT COUNT(*) as count FROM keyword_mentions WHERE tenant_id = ?',
        [defaultTenantId]
      );

      return {
        activeConversations: activeConversations[0].count,
        totalUsers: totalUsers[0].count,
        messagesToday: messagesToday[0].count,
        keywordsTracked: keywordsCount[0].count
      };
    } catch (error) {
      console.error('Error getting dashboard stats:', error);
      return {
        activeConversations: 0,
        totalUsers: 0,
        messagesToday: 0,
        keywordsTracked: 0
      };
    }
  }

  // Recent conversations
  async getRecentConversations(tenantId = null, limit = 10) {
    try {
      const defaultTenantId = tenantId || process.env.DEFAULT_TENANT_ID;
      const [rows] = await this.db.execute(
        `SELECT c.*, u.name as user_name, u.phone_number, u.chat_id,
         (SELECT content FROM messages WHERE conversation_id = c.id ORDER BY created_at DESC LIMIT 1) as last_message
         FROM conversations c 
         JOIN users u ON c.user_id = u.id 
         WHERE c.tenant_id = ? 
         ORDER BY c.last_message_at DESC 
         LIMIT ?`,
        [defaultTenantId, limit]
      );
      return rows;
    } catch (error) {
      console.error('Error getting recent conversations:', error);
      return [];
    }
  }

  // Admin authentication
  async getAdminUser(username) {
    try {
      const [rows] = await this.db.execute(
        'SELECT * FROM admin_users WHERE username = ?',
        [username]
      );
      return rows[0] || null;
    } catch (error) {
      console.error('Error getting admin user:', error);
      return null;
    }
  }

  async createAdminUser(username, password, email = null) {
    try {
      const hashedPassword = await bcrypt.hash(password, 10);
      const id = uuidv4();
      
      await this.db.execute(
        'INSERT INTO admin_users (id, username, password, email) VALUES (?, ?, ?, ?)',
        [id, username, hashedPassword, email]
      );
      
      return await this.getAdminUser(username);
    } catch (error) {
      console.error('Error creating admin user:', error);
      throw error;
    }
  }

  async authenticateAdmin(username, password) {
    try {
      const admin = await this.getAdminUser(username);
      if (!admin) {
        return null;
      }
      
      const isValid = await bcrypt.compare(password, admin.password);
      if (!isValid) {
        return null;
      }
      
      // Return admin without password
      const { password: _, ...adminWithoutPassword } = admin;
      return adminWithoutPassword;
    } catch (error) {
      console.error('Error authenticating admin:', error);
      return null;
    }
  }

  // Business templates
  async getBusinessTemplates() {
    try {
      const [rows] = await this.db.execute(
        'SELECT * FROM business_templates WHERE is_active = TRUE ORDER BY name'
      );
      return rows.map(row => ({
        ...row,
        prompts: row.prompts ? JSON.parse(row.prompts) : [],
        keywords: row.keywords ? JSON.parse(row.keywords) : []
      }));
    } catch (error) {
      console.error('Error getting business templates:', error);
      return [];
    }
  }

  // WhatsApp providers
  async getWhatsAppProvider(tenantId = null) {
    try {
      const defaultTenantId = tenantId || process.env.DEFAULT_TENANT_ID;
      const [rows] = await this.db.execute(
        'SELECT * FROM whatsapp_providers WHERE tenant_id = ? AND is_active = TRUE ORDER BY is_primary DESC LIMIT 1',
        [defaultTenantId]
      );
      return rows[0] || null;
    } catch (error) {
      console.error('Error getting WhatsApp provider:', error);
      return null;
    }
  }

  // File management
  async createFile(fileData) {
    try {
      const id = uuidv4();
      const defaultTenantId = fileData.tenant_id || process.env.DEFAULT_TENANT_ID;
      
      await this.db.execute(
        `INSERT INTO files (id, tenant_id, filename, original_name, file_path, file_size, mime_type, file_type, uploaded_by, metadata) 
         VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
        [
          id,
          defaultTenantId,
          fileData.filename,
          fileData.original_name,
          fileData.file_path,
          fileData.file_size,
          fileData.mime_type || null,
          fileData.file_type || 'other',
          fileData.uploaded_by || null,
          fileData.metadata ? JSON.stringify(fileData.metadata) : null
        ]
      );

      return id;
    } catch (error) {
      console.error('Error creating file record:', error);
      throw error;
    }
  }

  // System logs
  async logEvent(level, category, message, data = null, tenantId = null, userId = null) {
    try {
      const id = uuidv4();
      
      await this.db.execute(
        `INSERT INTO system_logs (id, tenant_id, user_id, level, category, message, data) 
         VALUES (?, ?, ?, ?, ?, ?, ?)`,
        [
          id,
          tenantId,
          userId,
          level,
          category,
          message,
          data ? JSON.stringify(data) : null
        ]
      );
    } catch (error) {
      console.error('Error logging event:', error);
    }
  }
}

export const storage = new DatabaseStorage();