import mysql from 'mysql2/promise';

export class DatabaseService {
  constructor() {
    this.connection = null;
  }

  async initialize() {
    try {
      this.connection = await mysql.createConnection({
        host: process.env.DB_HOST || 'localhost',
        user: process.env.DB_USER || 'root',
        password: process.env.DB_PASSWORD || '',
        database: process.env.DB_NAME || 'whatsapp_survey',
        charset: 'utf8mb4'
      });
      
      console.log('✅ Database connected successfully');
      
      // Create tables if they don't exist
      await this.createTables();
      
    } catch (error) {
      console.error('❌ Database connection failed:', error.message);
      throw error;
    }
  }

  async createTables() {
    const tables = [
      // Users table
      `CREATE TABLE IF NOT EXISTS users (
        id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
        username VARCHAR(255) UNIQUE NOT NULL,
        email VARCHAR(255) UNIQUE,
        password_hash VARCHAR(255) NOT NULL,
        role ENUM('admin', 'tenant', 'agency') DEFAULT 'tenant',
        tenant_id VARCHAR(36),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
      )`,
      
      // Tenants table
      `CREATE TABLE IF NOT EXISTS tenants (
        id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
        name VARCHAR(255) NOT NULL,
        domain VARCHAR(255),
        settings JSON,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
      )`,
      
      // Conversations table
      `CREATE TABLE IF NOT EXISTS conversations (
        id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
        tenant_id VARCHAR(36) NOT NULL,
        customer_phone VARCHAR(20) NOT NULL,
        customer_name VARCHAR(255),
        status ENUM('active', 'completed', 'pending') DEFAULT 'active',
        last_message_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (tenant_id) REFERENCES tenants(id)
      )`,
      
      // Messages table
      `CREATE TABLE IF NOT EXISTS messages (
        id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
        conversation_id VARCHAR(36) NOT NULL,
        sender ENUM('customer', 'bot') NOT NULL,
        content TEXT NOT NULL,
        message_type ENUM('text', 'image', 'document', 'audio') DEFAULT 'text',
        timestamp TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (conversation_id) REFERENCES conversations(id)
      )`,
      
      // Surveys table
      `CREATE TABLE IF NOT EXISTS surveys (
        id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
        tenant_id VARCHAR(36) NOT NULL,
        name VARCHAR(255) NOT NULL,
        description TEXT,
        questions JSON,
        active BOOLEAN DEFAULT true,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (tenant_id) REFERENCES tenants(id)
      )`,
      
      // Survey responses table
      `CREATE TABLE IF NOT EXISTS survey_responses (
        id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
        survey_id VARCHAR(36) NOT NULL,
        conversation_id VARCHAR(36) NOT NULL,
        responses JSON,
        demographic_data JSON,
        business_potential ENUM('high', 'medium', 'low') DEFAULT 'medium',
        completion_status ENUM('pending', 'partial', 'completed') DEFAULT 'pending',
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (survey_id) REFERENCES surveys(id),
        FOREIGN KEY (conversation_id) REFERENCES conversations(id)
      )`,
      
      // Keywords table
      `CREATE TABLE IF NOT EXISTS keywords (
        id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
        tenant_id VARCHAR(36) NOT NULL,
        keyword VARCHAR(255) NOT NULL,
        count INT DEFAULT 0,
        channel VARCHAR(50) DEFAULT 'whatsapp',
        last_mentioned TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (tenant_id) REFERENCES tenants(id)
      )`,
      
      // Business templates table
      `CREATE TABLE IF NOT EXISTS business_templates (
        id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
        name VARCHAR(255) NOT NULL,
        category VARCHAR(100),
        description TEXT,
        prompts JSON,
        keywords JSON,
        active BOOLEAN DEFAULT true,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )`
    ];

    for (const tableQuery of tables) {
      try {
        await this.connection.execute(tableQuery);
      } catch (error) {
        console.error('❌ Table creation error:', error.message);
      }
    }
    
    console.log('✅ Database tables initialized');
    
    // Seed initial data
    await this.seedInitialData();
  }

  async seedInitialData() {
    try {
      // Create default tenant
      const [tenantRows] = await this.connection.execute(
        'SELECT id FROM tenants WHERE name = ?',
        ['Default Tenant']
      );
      
      let tenantId;
      if (tenantRows.length === 0) {
        const [result] = await this.connection.execute(
          'INSERT INTO tenants (name, domain, settings) VALUES (?, ?, ?)',
          ['Default Tenant', 'infowasap.my', JSON.stringify({ 
            whatsapp: { enabled: true },
            ai: { enabled: true }
          })]
        );
        tenantId = result.insertId;
      } else {
        tenantId = tenantRows[0].id;
      }
      
      // Create admin user
      const [adminRows] = await this.connection.execute(
        'SELECT id FROM users WHERE username = ?',
        ['admin']
      );
      
      if (adminRows.length === 0) {
        const bcrypt = await import('bcryptjs');
        const hashedPassword = await bcrypt.hash('##Master8899', 12);
        
        await this.connection.execute(
          'INSERT INTO users (username, email, password_hash, role, tenant_id) VALUES (?, ?, ?, ?, ?)',
          ['admin', 'admin@infowasap.my', hashedPassword, 'admin', tenantId]
        );
        console.log('✅ Admin user created (username: admin, password: ##Master8899)');
      }
      
      // Seed business templates
      await this.seedBusinessTemplates();
      
    } catch (error) {
      console.error('❌ Seeding error:', error.message);
    }
  }

  async seedBusinessTemplates() {
    const templates = [
      {
        name: 'Air Conditioning Service',
        category: 'Home Services',
        description: 'Survey template for air conditioning repair and maintenance services',
        prompts: JSON.stringify({
          greeting: 'Hello! I\'m here to help you with air conditioning services. What type of AC service do you need?',
          followup: 'Can you tell me more about the issue you\'re experiencing?'
        }),
        keywords: JSON.stringify(['aircond', 'ac', 'air conditioning', 'cooling', 'repair', 'service'])
      },
      {
        name: 'Food Delivery Business',
        category: 'Food & Beverage',
        description: 'Assessment template for food delivery and catering businesses',
        prompts: JSON.stringify({
          greeting: 'Hi! I\'m conducting a survey about food delivery services in your area. Are you interested in food business opportunities?',
          followup: 'What type of food business are you considering?'
        }),
        keywords: JSON.stringify(['food', 'delivery', 'catering', 'restaurant', 'kitchen', 'cooking'])
      },
      {
        name: 'Online Business',
        category: 'E-commerce',
        description: 'Survey for assessing online business potential and digital literacy',
        prompts: JSON.stringify({
          greeting: 'Hello! I\'m here to learn about your interest in online business opportunities.',
          followup: 'Do you have experience with online selling or digital marketing?'
        }),
        keywords: JSON.stringify(['online', 'business', 'ecommerce', 'digital', 'internet', 'website'])
      }
    ];

    for (const template of templates) {
      try {
        const [rows] = await this.connection.execute(
          'SELECT id FROM business_templates WHERE name = ?',
          [template.name]
        );
        
        if (rows.length === 0) {
          await this.connection.execute(
            'INSERT INTO business_templates (name, category, description, prompts, keywords) VALUES (?, ?, ?, ?, ?)',
            [template.name, template.category, template.description, template.prompts, template.keywords]
          );
        }
      } catch (error) {
        console.error(`❌ Template seeding error for ${template.name}:`, error.message);
      }
    }
    
    console.log('✅ Business templates seeded');
  }

  async query(sql, params = []) {
    try {
      const [rows] = await this.connection.execute(sql, params);
      return rows;
    } catch (error) {
      console.error('❌ Database query error:', error.message);
      throw error;
    }
  }

  async close() {
    if (this.connection) {
      await this.connection.end();
    }
  }
}