import OpenAI from 'openai';

export class OpenAIService {
  constructor() {
    if (!process.env.OPENAI_API_KEY || process.env.OPENAI_API_KEY === 'sk-dummy-key-for-startup-only') {
      throw new Error('OpenAI API key not configured');
    }
    
    this.client = new OpenAI({
      apiKey: process.env.OPENAI_API_KEY
    });
  }

  async analyzeSurveyIntent(message, context = {}) {
    try {
      const prompt = `Analyze this customer message for business potential in the Malaysian context:

Message: "${message}"
Context: ${JSON.stringify(context)}

Assess the following:
1. Business interest level (high/medium/low)
2. Urgency of need (urgent/moderate/low)
3. Financial capability indicators
4. Business category they're interested in
5. Demographic hints (age group, occupation, location)

Respond in JSON format:
{
  "businessPotential": "high|medium|low",
  "urgency": "urgent|moderate|low", 
  "category": "business category",
  "demographics": {
    "ageGroup": "estimated age group",
    "occupation": "likely occupation",
    "location": "mentioned location"
  },
  "nextAction": "recommended follow-up action",
  "confidence": 0.85
}`;

      const response = await this.client.chat.completions.create({
        model: "gpt-4o", // Latest OpenAI model
        messages: [
          {
            role: "system",
            content: "You are an expert business analyst specializing in Malaysian small business assessment. Analyze customer messages to determine business potential and provide actionable insights."
          },
          {
            role: "user",
            content: prompt
          }
        ],
        response_format: { type: "json_object" },
        temperature: 0.3
      });

      return JSON.parse(response.choices[0].message.content);
    } catch (error) {
      console.error('❌ OpenAI analysis error:', error.message);
      return {
        businessPotential: 'medium',
        urgency: 'moderate',
        category: 'general',
        demographics: {},
        nextAction: 'continue conversation',
        confidence: 0.1
      };
    }
  }

  async generateResponse(message, context = {}) {
    try {
      const prompt = `Generate a helpful response for this WhatsApp customer:

Customer message: "${message}"
Business context: ${JSON.stringify(context)}
Language: Malaysian English/Malay mix (as appropriate)

Guidelines:
- Be friendly and professional
- Ask relevant follow-up questions
- Show genuine interest in their business needs
- Keep responses concise (WhatsApp style)
- Use appropriate Malaysian context and terms

Generate a natural, helpful response:`;

      const response = await this.client.chat.completions.create({
        model: "gpt-4o",
        messages: [
          {
            role: "system", 
            content: "You are a helpful business assistant for Malaysian customers via WhatsApp. Be friendly, professional, and culturally appropriate."
          },
          {
            role: "user",
            content: prompt
          }
        ],
        temperature: 0.7,
        max_tokens: 200
      });

      return response.choices[0].message.content.trim();
    } catch (error) {
      console.error('❌ OpenAI response generation error:', error.message);
      return "Terima kasih for your message! Our team will get back to you soon. 😊";
    }
  }

  async extractDemographics(conversation) {
    try {
      const prompt = `Extract demographic information from this conversation:

${conversation}

Extract the following if mentioned:
- Age or age group
- Gender
- Race/ethnicity
- Occupation
- Location (state/city)
- Income level indicators
- Business experience
- Education level

Return JSON format:
{
  "age": "age or age group",
  "gender": "male/female/not specified",
  "race": "malay/chinese/indian/other/not specified",
  "occupation": "current occupation",
  "location": "location mentioned",
  "incomeLevel": "high/medium/low/not specified",
  "businessExperience": "experienced/beginner/none/not specified",
  "education": "education level if mentioned"
}`;

      const response = await this.client.chat.completions.create({
        model: "gpt-4o",
        messages: [
          {
            role: "system",
            content: "You are a demographic data extraction specialist. Extract only explicitly mentioned information, never assume or infer."
          },
          {
            role: "user", 
            content: prompt
          }
        ],
        response_format: { type: "json_object" },
        temperature: 0.1
      });

      return JSON.parse(response.choices[0].message.content);
    } catch (error) {
      console.error('❌ Demographic extraction error:', error.message);
      return {};
    }
  }

  async categorizeKeywords(text) {
    try {
      const response = await this.client.chat.completions.create({
        model: "gpt-4o",
        messages: [
          {
            role: "system",
            content: "Extract important business-related keywords from the text. Return as a JSON array of keywords."
          },
          {
            role: "user",
            content: `Extract keywords from: "${text}"`
          }
        ],
        response_format: { type: "json_object" },
        temperature: 0.2
      });

      const result = JSON.parse(response.choices[0].message.content);
      return result.keywords || [];
    } catch (error) {
      console.error('❌ Keyword extraction error:', error.message);
      return [];
    }
  }
}