export class WhatsAppService {
  constructor() {
    this.instanceId = process.env.GREEN_API_INSTANCE_ID;
    this.accessToken = process.env.GREEN_API_ACCESS_TOKEN;
    this.baseUrl = `https://7103.api.greenapi.com/waInstance${this.instanceId}`;
    
    if (!this.instanceId || !this.accessToken) {
      throw new Error('WhatsApp credentials not configured');
    }
  }

  async initialize() {
    try {
      // Test connection
      const response = await fetch(`${this.baseUrl}/getStateInstance/${this.accessToken}`);
      const data = await response.json();
      
      if (data.stateInstance === 'authorized') {
        console.log('✅ WhatsApp API connected and authorized');
        
        // Start message polling
        this.startPolling();
        
        return true;
      } else {
        console.log('⚠️ WhatsApp API not authorized. Please scan QR code.');
        return false;
      }
    } catch (error) {
      console.error('❌ WhatsApp initialization error:', error.message);
      throw error;
    }
  }

  async sendMessage(chatId, message, options = {}) {
    try {
      const endpoint = `${this.baseUrl}/sendMessage/${this.accessToken}`;
      
      const payload = {
        chatId: chatId,
        message: message,
        ...options
      };

      const response = await fetch(endpoint, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify(payload)
      });

      const result = await response.json();
      
      if (result.idMessage) {
        console.log(`✅ Message sent to ${chatId}: ${message.substring(0, 50)}...`);
        return result;
      } else {
        console.error('❌ Failed to send message:', result);
        return null;
      }
    } catch (error) {
      console.error('❌ Send message error:', error.message);
      return null;
    }
  }

  async sendButtons(chatId, message, buttons) {
    try {
      const endpoint = `${this.baseUrl}/sendButtons/${this.accessToken}`;
      
      const payload = {
        chatId: chatId,
        message: message,
        footer: "WhatsApp Survey Bot",
        buttons: buttons.map((btn, index) => ({
          buttonId: `btn_${index}`,
          buttonText: { displayText: btn }
        }))
      };

      const response = await fetch(endpoint, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify(payload)
      });

      const result = await response.json();
      console.log(`✅ Buttons sent to ${chatId}`);
      return result;
    } catch (error) {
      console.error('❌ Send buttons error:', error.message);
      return null;
    }
  }

  async startPolling() {
    console.log('🔄 Starting WhatsApp message polling...');
    
    const poll = async () => {
      try {
        const response = await fetch(`${this.baseUrl}/receiveNotification/${this.accessToken}`);
        
        if (response.ok) {
          const notification = await response.json();
          
          if (notification && notification.body) {
            await this.handleIncomingMessage(notification);
            
            // Delete the notification after processing
            await fetch(`${this.baseUrl}/deleteNotification/${this.accessToken}/${notification.receiptId}`, {
              method: 'DELETE'
            });
          }
        }
      } catch (error) {
        console.error('❌ Polling error:', error.message);
      }
    };

    // Poll every 5 seconds
    setInterval(poll, 5000);
    console.log('✅ WhatsApp polling started (5s interval)');
  }

  async handleIncomingMessage(notification) {
    try {
      const messageData = notification.body;
      
      // Skip if it's our own message
      if (messageData.senderData?.sender === this.instanceId) {
        return;
      }

      const chatId = messageData.senderData?.chatId || messageData.chatId;
      const messageText = messageData.messageData?.textMessageData?.textMessage || '';
      const senderName = messageData.senderData?.senderName || 'Unknown';

      if (chatId && messageText) {
        console.log(`📩 New message from ${senderName}: ${messageText}`);
        
        // Process the message (integrate with your business logic)
        await this.processIncomingMessage({
          chatId,
          message: messageText,
          senderName,
          timestamp: new Date(),
          messageData
        });
      }
    } catch (error) {
      console.error('❌ Message handling error:', error.message);
    }
  }

  async processIncomingMessage(data) {
    try {
      // Here you would integrate with your business logic:
      // 1. Check if this is a new or existing conversation
      // 2. Analyze the message intent using OpenAI
      // 3. Generate appropriate response
      // 4. Update conversation state
      // 5. Send response back to customer

      const { chatId, message, senderName } = data;
      
      // Simple auto-response for now
      const responses = [
        "Terima kasih for your message! Our team will respond soon. 😊",
        "Hello! Thank you for contacting us. How can we help you today?",
        "Hi there! We received your message and will get back to you shortly.",
        "Selamat datang! Our business survey team will contact you soon."
      ];
      
      const randomResponse = responses[Math.floor(Math.random() * responses.length)];
      
      // Send response after a short delay to seem natural
      setTimeout(async () => {
        await this.sendMessage(chatId, randomResponse);
      }, 2000);
      
      console.log(`✅ Processed message from ${senderName}`);
    } catch (error) {
      console.error('❌ Message processing error:', error.message);
    }
  }

  async handleWebhook(webhookData) {
    try {
      console.log('📨 Webhook received:', JSON.stringify(webhookData, null, 2));
      
      if (webhookData.typeWebhook === 'incomingMessageReceived') {
        await this.handleIncomingMessage(webhookData);
      }
      
      return true;
    } catch (error) {
      console.error('❌ Webhook handling error:', error.message);
      return false;
    }
  }

  async getQRCode() {
    try {
      const response = await fetch(`${this.baseUrl}/qr/${this.accessToken}`);
      const data = await response.json();
      return data;
    } catch (error) {
      console.error('❌ QR code error:', error.message);
      return null;
    }
  }

  async getAccountInfo() {
    try {
      const response = await fetch(`${this.baseUrl}/getWaSettings/${this.accessToken}`);
      const data = await response.json();
      return data;
    } catch (error) {
      console.error('❌ Account info error:', error.message);
      return null;
    }
  }
}