// Complete database schema for MySQL/MariaDB - Full Feature Set
export const createTables = `
-- Sessions table (required for auth)
CREATE TABLE IF NOT EXISTS sessions (
  session_id VARCHAR(128) PRIMARY KEY,
  expires INT UNSIGNED NOT NULL,
  data MEDIUMTEXT,
  INDEX idx_expires (expires)
);

-- Admin users table
CREATE TABLE IF NOT EXISTS admin_users (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  username VARCHAR(50) UNIQUE NOT NULL,
  password VARCHAR(255) NOT NULL,
  email VARCHAR(100),
  is_super_admin BOOLEAN DEFAULT FALSE,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Tenants table (multi-tenant support)
CREATE TABLE IF NOT EXISTS tenants (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  name VARCHAR(255) NOT NULL,
  slug VARCHAR(100) UNIQUE NOT NULL,
  description TEXT,
  contact_email VARCHAR(100),
  contact_phone VARCHAR(20),
  website VARCHAR(255),
  logo_url VARCHAR(255),
  is_active BOOLEAN DEFAULT TRUE,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Tenant users table
CREATE TABLE IF NOT EXISTS tenant_users (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  tenant_id VARCHAR(36) NOT NULL,
  username VARCHAR(50) NOT NULL,
  password VARCHAR(255) NOT NULL,
  email VARCHAR(100),
  role ENUM('admin', 'manager', 'viewer') DEFAULT 'viewer',
  is_active BOOLEAN DEFAULT TRUE,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  FOREIGN KEY (tenant_id) REFERENCES tenants(id) ON DELETE CASCADE,
  UNIQUE KEY unique_tenant_username (tenant_id, username)
);

-- WhatsApp users table
CREATE TABLE IF NOT EXISTS users (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  tenant_id VARCHAR(36) NOT NULL,
  phone_number VARCHAR(20) NOT NULL,
  chat_id VARCHAR(50) NOT NULL,
  name VARCHAR(255),
  profile_name VARCHAR(255),
  language ENUM('ms', 'en', 'zh', 'ta') DEFAULT 'ms',
  is_active BOOLEAN DEFAULT TRUE,
  last_activity TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  FOREIGN KEY (tenant_id) REFERENCES tenants(id) ON DELETE CASCADE,
  UNIQUE KEY unique_tenant_chat (tenant_id, chat_id),
  INDEX idx_phone_number (phone_number),
  INDEX idx_chat_id (chat_id),
  INDEX idx_last_activity (last_activity)
);

-- Surveys table
CREATE TABLE IF NOT EXISTS surveys (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  tenant_id VARCHAR(36) NOT NULL,
  title VARCHAR(255) NOT NULL,
  description TEXT,
  is_active BOOLEAN DEFAULT TRUE,
  created_by VARCHAR(36),
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  FOREIGN KEY (tenant_id) REFERENCES tenants(id) ON DELETE CASCADE,
  INDEX idx_tenant_active (tenant_id, is_active)
);

-- Conversations table
CREATE TABLE IF NOT EXISTS conversations (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  user_id VARCHAR(36) NOT NULL,
  tenant_id VARCHAR(36) NOT NULL,
  survey_id VARCHAR(36),
  title VARCHAR(255),
  status ENUM('active', 'inactive', 'archived') DEFAULT 'active',
  language ENUM('ms', 'en', 'zh', 'ta') DEFAULT 'ms',
  last_message_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
  FOREIGN KEY (tenant_id) REFERENCES tenants(id) ON DELETE CASCADE,
  FOREIGN KEY (survey_id) REFERENCES surveys(id) ON DELETE SET NULL,
  INDEX idx_user_status (user_id, status),
  INDEX idx_tenant_status (tenant_id, status),
  INDEX idx_last_message (last_message_at)
);

-- Messages table
CREATE TABLE IF NOT EXISTS messages (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  conversation_id VARCHAR(36) NOT NULL,
  content TEXT NOT NULL,
  sender_type ENUM('user', 'bot') NOT NULL,
  message_type ENUM('text', 'image', 'document', 'interactive') DEFAULT 'text',
  metadata JSON,
  processed_at TIMESTAMP NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (conversation_id) REFERENCES conversations(id) ON DELETE CASCADE,
  INDEX idx_conversation_type (conversation_id, sender_type),
  INDEX idx_created_at (created_at)
);

-- Survey responses table
CREATE TABLE IF NOT EXISTS survey_responses (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  conversation_id VARCHAR(36) NOT NULL,
  survey_id VARCHAR(36) NOT NULL,
  user_id VARCHAR(36) NOT NULL,
  tenant_id VARCHAR(36) NOT NULL,
  intent_category ENUM('high_potential', 'medium_potential', 'low_potential', 'not_eligible', 'unclear') DEFAULT 'unclear',
  business_interest_score INT DEFAULT 0,
  urgency_level ENUM('immediate', 'soon', 'future', 'none') DEFAULT 'none',
  eligibility_status ENUM('eligible', 'maybe_eligible', 'not_eligible', 'unknown') DEFAULT 'unknown',
  response_data JSON,
  analyzed_at TIMESTAMP NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  FOREIGN KEY (conversation_id) REFERENCES conversations(id) ON DELETE CASCADE,
  FOREIGN KEY (survey_id) REFERENCES surveys(id) ON DELETE CASCADE,
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
  FOREIGN KEY (tenant_id) REFERENCES tenants(id) ON DELETE CASCADE,
  INDEX idx_tenant_category (tenant_id, intent_category),
  INDEX idx_survey_status (survey_id, eligibility_status)
);

-- Demographics table
CREATE TABLE IF NOT EXISTS demographics (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  user_id VARCHAR(36) NOT NULL,
  tenant_id VARCHAR(36) NOT NULL,
  age_range VARCHAR(20),
  gender ENUM('male', 'female', 'other', 'unknown') DEFAULT 'unknown',
  race VARCHAR(50),
  occupation VARCHAR(100),
  location VARCHAR(100),
  income_range VARCHAR(50),
  education_level VARCHAR(50),
  family_size INT,
  extracted_from_messages BOOLEAN DEFAULT TRUE,
  confidence_score DECIMAL(3,2) DEFAULT 0.00,
  last_updated TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
  FOREIGN KEY (tenant_id) REFERENCES tenants(id) ON DELETE CASCADE,
  UNIQUE KEY unique_user_demographics (user_id),
  INDEX idx_tenant_demographics (tenant_id),
  INDEX idx_age_gender (age_range, gender)
);

-- Keywords table
CREATE TABLE IF NOT EXISTS keywords (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  tenant_id VARCHAR(36) NOT NULL,
  survey_id VARCHAR(36),
  keyword VARCHAR(255) NOT NULL,
  category ENUM('product', 'service', 'support', 'pricing', 'general', 'complaint', 'compliment') DEFAULT 'general',
  response_text TEXT,
  language ENUM('ms', 'en', 'zh', 'ta') DEFAULT 'ms',
  is_active BOOLEAN DEFAULT TRUE,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  FOREIGN KEY (tenant_id) REFERENCES tenants(id) ON DELETE CASCADE,
  FOREIGN KEY (survey_id) REFERENCES surveys(id) ON DELETE SET NULL,
  UNIQUE KEY unique_tenant_keyword (tenant_id, keyword, language),
  INDEX idx_tenant_category (tenant_id, category),
  INDEX idx_keyword_search (keyword)
);

-- Keyword mentions table
CREATE TABLE IF NOT EXISTS keyword_mentions (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  tenant_id VARCHAR(36) NOT NULL,
  survey_id VARCHAR(36),
  keyword VARCHAR(255) NOT NULL,
  count INT DEFAULT 1,
  channel ENUM('whatsapp', 'telegram', 'web') DEFAULT 'whatsapp',
  last_mentioned TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (tenant_id) REFERENCES tenants(id) ON DELETE CASCADE,
  FOREIGN KEY (survey_id) REFERENCES surveys(id) ON DELETE SET NULL,
  INDEX idx_tenant_keyword (tenant_id, keyword),
  INDEX idx_count_desc (count DESC),
  INDEX idx_last_mentioned (last_mentioned)
);

-- Customer leads table
CREATE TABLE IF NOT EXISTS customer_leads (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  tenant_id VARCHAR(36) NOT NULL,
  user_id VARCHAR(36) NOT NULL,
  survey_id VARCHAR(36),
  lead_score INT DEFAULT 0,
  status ENUM('new', 'contacted', 'qualified', 'converted', 'lost') DEFAULT 'new',
  source VARCHAR(50) DEFAULT 'whatsapp',
  contact_preference ENUM('whatsapp', 'call', 'email') DEFAULT 'whatsapp',
  notes TEXT,
  follow_up_date DATE,
  assigned_to VARCHAR(36),
  converted_at TIMESTAMP NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  FOREIGN KEY (tenant_id) REFERENCES tenants(id) ON DELETE CASCADE,
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
  FOREIGN KEY (survey_id) REFERENCES surveys(id) ON DELETE SET NULL,
  INDEX idx_tenant_status (tenant_id, status),
  INDEX idx_lead_score (lead_score DESC),
  INDEX idx_follow_up (follow_up_date)
);

-- Business templates table
CREATE TABLE IF NOT EXISTS business_templates (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  name VARCHAR(255) NOT NULL,
  category VARCHAR(100) NOT NULL,
  description TEXT,
  icon VARCHAR(10),
  prompts JSON,
  keywords JSON,
  is_active BOOLEAN DEFAULT TRUE,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  UNIQUE KEY unique_template_name (name),
  INDEX idx_category (category)
);

-- WhatsApp providers table
CREATE TABLE IF NOT EXISTS whatsapp_providers (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  tenant_id VARCHAR(36) NOT NULL,
  provider_name ENUM('green_api', 'getitsms', 'meta_api') NOT NULL,
  instance_id VARCHAR(100),
  access_token VARCHAR(255),
  webhook_url VARCHAR(255),
  is_active BOOLEAN DEFAULT FALSE,
  is_primary BOOLEAN DEFAULT FALSE,
  settings JSON,
  last_health_check TIMESTAMP NULL,
  health_status ENUM('healthy', 'warning', 'error') DEFAULT 'healthy',
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  FOREIGN KEY (tenant_id) REFERENCES tenants(id) ON DELETE CASCADE,
  UNIQUE KEY unique_tenant_provider (tenant_id, provider_name),
  INDEX idx_tenant_active (tenant_id, is_active)
);

-- Pricing plans table
CREATE TABLE IF NOT EXISTS pricing_plans (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  name VARCHAR(100) NOT NULL,
  description TEXT,
  price_monthly DECIMAL(10,2) DEFAULT 0.00,
  price_yearly DECIMAL(10,2) DEFAULT 0.00,
  max_messages INT DEFAULT 1000,
  max_users INT DEFAULT 100,
  max_surveys INT DEFAULT 1,
  features JSON,
  is_active BOOLEAN DEFAULT TRUE,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  UNIQUE KEY unique_plan_name (name)
);

-- Payment gateways table
CREATE TABLE IF NOT EXISTS payment_gateways (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  name VARCHAR(100) NOT NULL,
  provider VARCHAR(50) NOT NULL,
  api_key VARCHAR(255),
  secret_key VARCHAR(255),
  webhook_url VARCHAR(255),
  is_active BOOLEAN DEFAULT FALSE,
  is_sandbox BOOLEAN DEFAULT TRUE,
  supported_methods JSON,
  settings JSON,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  UNIQUE KEY unique_gateway_name (name)
);

-- Tenant subscriptions table
CREATE TABLE IF NOT EXISTS tenant_subscriptions (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  tenant_id VARCHAR(36) NOT NULL,
  pricing_plan_id VARCHAR(36) NOT NULL,
  status ENUM('active', 'cancelled', 'expired', 'pending') DEFAULT 'pending',
  starts_at TIMESTAMP NOT NULL,
  expires_at TIMESTAMP NOT NULL,
  payment_gateway_id VARCHAR(36),
  subscription_data JSON,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  FOREIGN KEY (tenant_id) REFERENCES tenants(id) ON DELETE CASCADE,
  FOREIGN KEY (pricing_plan_id) REFERENCES pricing_plans(id) ON DELETE RESTRICT,
  FOREIGN KEY (payment_gateway_id) REFERENCES payment_gateways(id) ON DELETE SET NULL,
  INDEX idx_tenant_status (tenant_id, status),
  INDEX idx_expiry (expires_at)
);

-- Files table (for document management)
CREATE TABLE IF NOT EXISTS files (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  tenant_id VARCHAR(36) NOT NULL,
  filename VARCHAR(255) NOT NULL,
  original_name VARCHAR(255) NOT NULL,
  file_path VARCHAR(500) NOT NULL,
  file_size BIGINT NOT NULL,
  mime_type VARCHAR(100),
  file_type ENUM('image', 'document', 'audio', 'video', 'other') DEFAULT 'other',
  uploaded_by VARCHAR(36),
  is_processed BOOLEAN DEFAULT FALSE,
  processed_at TIMESTAMP NULL,
  metadata JSON,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (tenant_id) REFERENCES tenants(id) ON DELETE CASCADE,
  INDEX idx_tenant_type (tenant_id, file_type),
  INDEX idx_uploaded_by (uploaded_by)
);

-- System logs table
CREATE TABLE IF NOT EXISTS system_logs (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  tenant_id VARCHAR(36),
  user_id VARCHAR(36),
  level ENUM('info', 'warning', 'error', 'debug') DEFAULT 'info',
  category VARCHAR(50) NOT NULL,
  message TEXT NOT NULL,
  data JSON,
  ip_address VARCHAR(45),
  user_agent TEXT,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (tenant_id) REFERENCES tenants(id) ON DELETE SET NULL,
  FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE SET NULL,
  INDEX idx_tenant_level (tenant_id, level),
  INDEX idx_category_created (category, created_at),
  INDEX idx_created_at (created_at)
);

-- System settings table
CREATE TABLE IF NOT EXISTS system_settings (
  id VARCHAR(36) PRIMARY KEY DEFAULT (UUID()),
  setting_key VARCHAR(100) UNIQUE NOT NULL,
  setting_value TEXT,
  description TEXT,
  is_public BOOLEAN DEFAULT FALSE,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  INDEX idx_key_public (setting_key, is_public)
);

-- Insert default data
INSERT IGNORE INTO tenants (id, name, slug, description, contact_email, is_active) VALUES
('a1ffbb8c-cb66-43de-8d04-175289124254', 'Default Tenant', 'default', 'Default system tenant', 'admin@system.com', TRUE);

INSERT IGNORE INTO admin_users (id, username, password, email, is_super_admin) VALUES
('admin-001', 'admin', '$2b$10$rQzpQKtJQqZqZpQZqZpQZez8z8z8z8z8z8z8z8z8z8z8z8z8z8z8', 'admin@system.com', TRUE);

INSERT IGNORE INTO pricing_plans (name, description, price_monthly, max_messages, max_users, max_surveys) VALUES
('Freemium', 'Basic plan with limited features', 0.00, 1000, 100, 1),
('Starter', 'Perfect for small businesses', 29.00, 5000, 500, 3),
('Growth', 'Ideal for growing companies', 99.00, 20000, 2000, 10),
('Enterprise', 'Full-featured plan for large organizations', 299.00, 100000, 10000, 50);

INSERT IGNORE INTO system_settings (setting_key, setting_value, description, is_public) VALUES
('app_name', 'WhatsApp Survey Chatbot', 'Application name', TRUE),
('app_version', '1.0.0', 'Application version', TRUE),
('maintenance_mode', 'false', 'Maintenance mode status', FALSE),
('max_file_size', '10485760', 'Maximum file upload size in bytes', FALSE);
`;

export default { createTables };